/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#ifndef _kernel_h_
#define _kernel_h_

#include <nano_limits.h>
#include <kalloc.h>
#include "_time.h"


#ifdef DLOG_SUPPORT
    void SetSysTimerScale(long period_ns);
#endif


/* Circular Queue structure */
/* Used to hook things together. */
#ifndef _queue_struct
#define _queue_struct
typedef struct Gqueue
{
    struct Gqueue *q_head;
    struct Gqueue *q_tail;
} queue;
#endif


/* td_replyq must be right above the td_sendq's in the TD because there are
   times (During initialization and destruction) when it is convenient to
   address td_replyq as td_sendq[-1] */
typedef struct
{
    BitMask td_portmask;
    BitMask td_portpend;
    int td_nports;
    int td_lastport;
    struct Gqueue td_replyq;       /* Must be second last */
    struct Gqueue td_sendq[1];     /* Must be last */
} RendezDescriptor;

/* The size and format of the following structure is known by the assembler
   files. */
/* IKM 2007 -- Removed the shared fields from the head of the structure.
   There is now a full TMCB at the end, if the _POSIX_TIMEOUTS option is
   enabled. Modifications to the mr_xxx functions, which used the thread
   descriptor as a pseudo timer for timeouts. Modifications to the Posix
   timeoutable functions, which used a single timer in the thing being waited
   on -- not correct according to Posix 2001/2004 semantics. */
/* mr_xxx functions came from the Unison ur_xxx functions. These used the
   thread descriptor as a pseudo timer. So the thread lived on the timer
   queue, and the other ur_xxx functions would explicitly wake a waiting
   thread. (ur_send would wake a task sleeping in ur_receive, ur_reply would
   wake a task sleeping in ur_send) The woken thread would then have to
   figure out whether it timed out or completed. The waking thread modified
   the woken thread's td_state to pass the message. */
/* Posix timeout functions like mq_timedreceive() had a timer built into the
   object being waited on. This means that if a second task comes along and
   wants to do a timedreceive, it overrides the timer settings, so the
   previous task that was waiting with timeout loses its timeout, and waits
   forever. This violates Posix 2001/2004 semantics. (IKM -- It's not clear
   to me what semantics existed when SAM did this in 1995/6 or even if any of
   this was specified yet. He was working from dox generously donated by
   friends from Sun etc, and from hearsay from members of the IEEE Posix
   working group...) */
/* So... Now, each td has a full TMCB slapped on the end. (It is possible to
   use less space, but that requires adding more special-case code to
   timer_tick() and friends). When a timeoutable function is called, it
   places the thread on the appropriate wait queue (these have to be sorted
   by priority under Posix rules), and fills the timer to be placed on the
   timer queue, and sleeps. On waking up, if the state is TIMER_EXPIRED, then
   the timer expired, and we can report failure. If the state isn't, then the
   waited-for event happened, we can disarm the timer, and continue to
   process whatever it is we're processing. */
typedef struct Thread
{
    struct Gqueue td_mainq;        /* The queue as always goes at the head of
                                   the structure */
    /* Put items known by assembler here, so they don't move around based on
     other code inclusion options */
    char *td_stackpointer;        /* The thread SP saved when inactive */
#if defined(__PIC24F__) || defined(__dsPIC30F__) || defined(__dsPIC33F__)
    char *td_SPLIM;               /* This value is required for the stack
                                   checking hardware in the PIC30 family */
#endif
    pthread_t td_id;
    int td_state;                 /* See below for list of states */
    struct Gqueue *td_myqueue;    /* The thread's current priority -- the queue
                                   it's sitting on */
#if _POSIX_THREAD_PRIO_INHERIT > 0
    int td_priority;              /* The thread's assigned priority. If we're
                                   holding a thread mutex, we can see our
                                   effective priority (td_myqueue) boosted
                                   under the Posix thread priority
                                   inheritance rules. This holds our assigned
                                   priority so we can drop back down when we
                                   release the mutex. */

#endif                          /* _POSIX_THREAD_PRIO_INHERIT */
    int td_errno;

    /* td_stackbase points to the start of the allocated portion of the stack
     (i.e. returned from malloc). */
    char *td_stackbase;
    size_t td_stacksize;        /* for debugging */
    THREAD (*td_rootfunc) (void *arg);   /* for debugging */

    /* for sigpost */
    BitMask td_sigmask;
    BitMask td_sigpend;

    /* for mr_send */
    pthread_t td_correspondent;
    int td_sendport;
    NetAddr td_rqst;
    NetAddr td_rply;

    /* for mr_receive */
    RendezDescriptor *td_rd;

#ifdef MULTIPROC
    int td_nodeno;
#endif

#if _POSIX_TIMEOUTS > 0
    TMCB td_timeout;
#endif                          /* _POSIX_TIMEOUTS */

    /* for join and exit */
    struct Gqueue join_queue;     // for saving join-block thread
    void *exit_return;
    int detachstate;

    /* for select */
    unsigned long sel_flag_fd;

    /* for cancel */
    unsigned int cancelstatus;
    void *cleanup_stack;

#ifdef DLOG_SUPPORT
    uint_32 dlog_mask;
#endif
} Thread;

/* Thread states (td_state) */
#define READY           1       /* known in assembler */
#define SUSPENDED       2
#define DEAD            3
#define TIMER_EXPIRED   4
#define ARMED_TMCB      9       /* this is really an armed TMCB */
/* rendezvous states */
#define SEND_BLOCKED    10
#define RCV_BLOCKED     11
#define REPLY_BLOCKED   12
#define TIMER_BLOCKED   13
#define COPYING_MSG     14
#define SENDING         15      /* multiproc */
#define REPLYING        16      /* multiproc */
/* mqueue states */
#define MQ_BLOCKED      20
#define RESOURCE_BLK    21
#define MQ_TIMER        22
/* semaphore states */
#define SEM_BLOCKED     30
/* barrier states */
#define BARRIER_BLOCKED 40
/* join states */
#define JOIN_BLOCKED    50
/* cond var states */
#define COND_BLOCKED    60

extern Thread *volatile _Active;


typedef struct
{
    unsigned short size;
    unsigned short pad;
} MinHeader;
#define MSG_SIZE(m)    (((MinHeader *)(m))->size)


/* queue.c */
//void initqueue (struct Gqueue *ql); // use macros
#define initqueue(ql) do {(ql)->q_head = (ql); (ql)->q_tail = (ql);} while(0 == 1)
#define queueIsEmpty(ql) ((ql)->q_head == (ql))
void remqueue (struct Gqueue *td);
void insqueue (struct Gqueue *td, struct Gqueue *qh);
void insqueue_head (struct Gqueue *td, struct Gqueue *qh);
void insqueue_prio (struct Gqueue *td, struct Gqueue *qh);
void movequeue (struct Gqueue *td, struct Gqueue *qh);
int find_first_bit (BitMask mask);


/* thread.c */
void _Add_ready (Thread * td);
Thread *_Init_thread (char *stack, size_t stacksize,
                      int priority, void *arg, THREAD (*root_func) (void *arg));


/* These three routines handle the pthread_t/Thread* mapping. A default set
   exist in tdalloc.c */
Thread *TDalloc (int global);
void TDfree (Thread * td);
#ifndef MULTIPROC
/* For uniprocessor make this a macro for speed. */
#define CAST2TD(id)    ((Thread*)(id))
#define ConvertToTD(id) ((CAST2TD(id)->td_id == (id)) ? CAST2TD(id) : NULL)
#define mp_memcpy       memcpy
#define mp_int2ext(x)   (NetAddr)(x)
#define mp_ext2int(x)   (NetAddr)(x)
#define safe_copy       memcpy
#else
Thread *ConvertToTD (pthread_t id);
void *mp_memcpy(void *s1, const void *s2, size_t n);
NetAddr mp_int2ext (NetAddr addr);
NetAddr mp_ext2int (NetAddr addr);
void *safe_copy(void *s1, const void *s2, size_t n);
#endif


extern volatile unsigned int _locklevel;
extern volatile unsigned int _isrlevel;

extern struct Gqueue _Ready_q[PTHREAD_PRIO_MAX + 1];
extern struct Gqueue *prio_q;



#define IN_ISR()        (getisrlevel() != 0U)
#define RETURN_ERRNO(e) { if (IN_ISR()) return (e); else {errno = (e); return -1;} }


/* The following signals are used by the kernel code. Note that SIGMQMSG must
   be higher priority (lower number) than SIGTIMEOUT. */
#define SIGTIMEOUT        31
#define SIGMQMSG        30

/* internal functions, for kernel using */
int __pthread_mutex_lock (pthread_mutex_t *mp);



/*
 * Platform depended defines and functions
 */
/* Put at the base of a threads' stack so can do a crude check for stack overflow. */
#define STACK_MAGIC    0xaaaaaaaaU

void _Dispatch (void);
void _Relinquish (Status ps);
int platform_stackinfo (Thread *td, stackinfo_t *info);
char *platform_init_thread (Thread *td, void *arg, THREAD (*root_func) (void *arg));
void set_exit_addr(Thread *td);


/*
 * For port functions
 */
void StartFirstTask (void);
void ContextSwitch (void);


#endif /* _kernel_h_ */
