/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>

#if _POSIX_MESSAGE_PASSING > 0

#include "dlog_kernel_ev.h"
#include "_mqueue.h"
#include <stdarg.h>


#define EXCLUSIVE(oflag)    ((oflag) & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)

/* This is the default attributes for a message queue. It is global so that
   the user can modify it! */
struct mq_attr mqAttrDefault = { 0, 128, 1024, 0 };

/* Linked list of QCBs */
QCB *qHead = NULL;

static QD *create_qd(QCB * qcb, int mq_flags);

#define MQ_RETURN_ERRNO(e) { errno = (e); return (mqd_t)-1; }


/**
 * @addtogroup queue_group queue
 * @{
 */


/**
 *  @brief     Open a message queue.
 *
 *  @param[in] name: points to a string naming a message queue.
 *  @param[in] oflag: requests the desired receive and/or send access to the message queue.
 *  @param[in] args: list of parameters.
 *
 *  @retval    mqd_t: message queue descriptor.
 *  @retval    (mqd_t)(-1): error. Variable errno is set appropriately. (See errno.h)
 */
mqd_t mq_open(const char *name, int oflag, ...)
{
    QD *qd = NULL;                   /* must be null */
    QCB *qcb;
    Status ps;
    size_t len;

    len = strlen(name);
    if ((len > (size_t)NAME_MAX) || (len > (size_t)PATH_MAX))
    {
        MQ_RETURN_ERRNO(ENAMETOOLONG);
    }

    /* must specify at least read or write */
    if ((oflag & O_RDWR) == 0)
        MQ_RETURN_ERRNO(EINVAL);

    ps = i_disable();
    qcb = name2qcb(name, NULL);
    if (qcb != NULL)
    {                             /* queue exists */
        if ((qcb->attr.mq_flags & MQ_UNLINK) != 0)
        {
            i_restore(ps);
            MQ_RETURN_ERRNO(EACCES);
        }
        if (EXCLUSIVE(oflag))
        {
            i_restore(ps);
            MQ_RETURN_ERRNO(EEXIST);
        }
        /* create a QD that points to QCB */
        qd = create_qd(qcb, oflag);
        if (qd == NULL)
        {
            i_restore(ps);
            MQ_RETURN_ERRNO(ENOSPC);
        }
        ++qcb->ocount;
        goto qcreate_ok;
    }
    else if ((oflag & O_CREAT) != 0)
    {                             /* create it */
        size_t size;
        int maxmsg;
        struct mq_attr *attr;
        char *buf;
        va_list ap;
        mode_t varNotUsed;
        int flag;

        va_start(ap, oflag);

        /* Some compilers will give a warning about the following line not using
        the value. We must call the va_arg to skip the parameter but we never
        need it! */
        varNotUsed = va_arg(ap, mode_t);
        (void)varNotUsed;   //prevent compiler warning

        attr = va_arg(ap, struct mq_attr *);

        va_end(ap);

        /* if attr is null, use defaults */
        if (attr == NULL)
            attr = &mqAttrDefault;

        else if ((attr->mq_maxmsg <= 0) || (attr->mq_msgsize <= 0))
        {
            i_restore(ps);
            MQ_RETURN_ERRNO(EINVAL);
        }

        /* allocate the qcb */
        qcb = kalloc (KA_QCB, sizeof (QCB) + strlen (name));
        if (qcb == NULL)
            goto qcreate_failed;

        /* If user specified O_NONBLOCK in attr, use it */
        if ((attr->mq_flags & O_NONBLOCK) != 0)
            flag = oflag | O_NONBLOCK;
        else
            flag = oflag;

        /* allocate the queue descriptor */
        qd = create_qd (qcb, flag);
        if (qd == NULL)
            goto qcreate_failed;

        /* allocate the message buffers */
        /* TMS320 word */
        /*************** NEEDS ATTENTION PIC ERROR*********************/
        size = (size_t)ALIGN_LONG ((size_t)attr->mq_msgsize + sizeof (MsgQueue));
        buf = kalloc (KA_MSGB, size * (size_t)attr->mq_maxmsg);
        if (buf == NULL)
            goto qcreate_failed;
        qcb->msg = buf;

        /* init qcb */
        qcb->ocount = 1;
        (void)memcpy(&qcb->attr, attr, sizeof (struct mq_attr));
        (void)strcpy(qcb->name, name);
        qcb->attr.mq_curmsgs = 0;   /* always start at 0 */
        initqueue(&qcb->mQueue);
        initqueue(&qcb->tQueue);
        initqueue(&qcb->rQueue);
        initqueue(&qcb->fQueue);

        /* put messages on freelist */
        for (maxmsg = (int)attr->mq_maxmsg; maxmsg > 0; --maxmsg)
        {
            insqueue((struct Gqueue *) buf, &qcb->fQueue);
            buf += size;
        }

        /* link it in */
        qcb->next = qHead;
        qHead = qcb;
        goto qcreate_ok;

        qcreate_failed:
        i_restore(ps);
        if (qcb != NULL)
        {
            kfree(KA_QCB, qcb);
            if (qd != NULL)
                kfree(KA_QD, qd);
        }
        MQ_RETURN_ERRNO(ENOSPC);
    }

    /* queue does not exist */
    i_restore(ps);
    MQ_RETURN_ERRNO(ENOENT);

    qcreate_ok:
    i_restore(ps);

    // Log this kernel function
    DLOG_MQ_OPEN_INSERT;

    return (mqd_t)qd;
}


/**
 * @} end queue group
 */


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


/**
 *  @brief     Create message queue.
 *
 *  @param[in] qcb: Description for qcb
 *  @param[in] mq_flags: requests the desired receive and/or send access to the message queue
 *
 *  @retval    mqd_t: message queue descriptor.
 *  @retval    0: error, no available memory.
 */
static QD *create_qd(QCB *qcb, int mq_flags)
{
    QD *qd;

    qd = kalloc (KA_QD, sizeof (QD));
    if (qd == NULL)
        return NULL;

    qd->qcb = qcb;
    qd->mq_flags = mq_flags;
    qd->id = (uint)qd;

    return qd;
}


/**
 *  @brief     Find existed queue.
 *
 *  @param[in] name: points to a string naming a message queue.
 *  @param[out] prevp: pointer to previous message queue descriptor.
 *
 *  @retval    0: no available memory.
 *  @retval    qcb: pointer to existed message queue descriptor.
 */
QCB *name2qcb(const char *name, QCB **prevp)
{
    register QCB *qcb, *prev;

    prev = NULL;
    for (qcb = qHead; qcb != NULL; qcb = qcb->next)
    {
        if (strcmp(qcb->name, name) == 0)
        {
            if (prevp != NULL)
                *prevp = prev;
            return qcb;
        }

        prev = qcb;
    }

    return NULL;
}

/**
 * @} end private kernel functions group
 */

#endif /* _POSIX_MESSAGE_PASSING */
