/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>

#if _POSIX_MESSAGE_PASSING > 0

#include "dlog_kernel_ev.h"
#include "_mqueue.h"
#include "_time.h"
#include "_pthread.h"


/**
 * @addtogroup queue_group queue
 * @{
 */


/**
 *  @brief     Send a message to a message queue with timeout.
 *
 *  @param[in] mqdes: message queue descriptor.
 *  @param[in] msg_ptr: pointer to the send message.
 *  @param[in] msg_len: length of the message pointed by msg_ptr.
 *  @param[in] msg_prio: indicates how the message is inserted into the message queue.
 *  @param[in] timeout: the time interval.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int mq_timedsend(mqd_t mqdes, const char *msg_ptr, size_t msg_len,
              unsigned int msg_prio, const struct timespec *timeout)
{
    Status ps;
    QCB *qcb;
    char *msg;
    int oldtype;

    /* Check for cancellation */
    CANCELLATION_POINT();

    if (!VALID_MQUEUE(mqdes))
        RETURN_ERRNO(EBADF);
    qcb = qd2qcb (mqdes);
    if (qcb == NULL)
        RETURN_ERRNO(EBADF);
    if (SENDOK(mqdes) == 0)
        RETURN_ERRNO(EBADF);
    if (msg_len > (size_t)qcb->attr.mq_msgsize)
        RETURN_ERRNO(EMSGSIZE);
    if (msg_prio >= (unsigned int)MQ_PRIO_MAX)
        RETURN_ERRNO(EINVAL);    // The value of msg_prio shall be less than {MQ_PRIO_MAX}.

    // Log this kernel function
    DLOG_MQ_SEND_INSERT;

    /* get a message buffer */
    ps = i_disable();
    while (queueIsEmpty(&qcb->fQueue))
    {
        if ((((QD *) mqdes)->mq_flags & O_NONBLOCK) != 0)
        {
            i_restore(ps);
            RETURN_ERRNO(EAGAIN);
        }

#if _POSIX_TIMEOUTS > 0
        if (timeout != NULL)
        {                           /* put on timer queue */
            struct itimerspec itimer;

            if ((timeout->tv_nsec < 0) || (timeout->tv_nsec >= 1000000000))
            {
                i_restore(ps);
                RETURN_ERRNO(EINVAL);
            }

            itimer.it_value.tv_sec = timeout->tv_sec;
            itimer.it_value.tv_nsec = timeout->tv_nsec;
            itimer.it_interval.tv_sec = 0;
            itimer.it_interval.tv_nsec = 0;
            __timer_settime((TMCB *)(&_Active->td_timeout), 0, &itimer, NULL);
        }
#endif /* _POSIX_TIMEOUTS */


        /* block and wait for free message buffer */
        _Active->td_state = RESOURCE_BLK;
        remqueue((struct Gqueue *)_Active);
        insqueue_prio((struct Gqueue *)_Active, &qcb->rQueue);
        __pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);    //set async type
        _Relinquish(ps);
        __pthread_setcanceltype(oldtype, NULL);    //restore old type
#if _POSIX_TIMEOUTS > 0
        if (_Active->td_state == TIMER_EXPIRED)
            RETURN_ERRNO(ETIMEDOUT);
#endif /* _POSIX_TIMEOUTS */
        ps = i_disable();
        /* Due to the interrupt window someone could beat us to the buffer. Loop
           again and be sure. */
    }

#if _POSIX_TIMEOUTS > 0
    if (timeout != NULL)
        /* disarm timer */
        __timer_settime((TMCB *)(&_Active->td_timeout), 0, NULL, NULL);
#endif /* _POSIX_TIMEOUTS */

    /* get the message from the queue */
    msg = (char *)qcb->fQueue.q_head;
    remqueue((struct Gqueue *)msg);

    /* Copy to the message buffer. Since we have removed the message from the
     queue of buffers, it is safe to enable here. */
    i_restore(ps);
    (void)mp_memcpy(msg + sizeof(MsgQueue), msg_ptr, msg_len);
    ((MsgQueue *)msg)->len = (int)msg_len;
    ((MsgQueue *)msg)->prio = msg_prio;
    ps = i_disable();

    /* Insert message in priority order, highest first. */
    {
        register struct Gqueue *qh;
        register struct Gqueue *queue;
        register struct Gqueue *qp, *ql;

        qh = (struct Gqueue *)&qcb->mQueue;
        queue = (struct Gqueue *)qh->q_tail;
        ql = (struct Gqueue *)msg;  // not so sure about this kr

        /* Optimization for lowest priority message */
        if (queueIsEmpty(qh) || (msg_prio <= ((MsgQueue *)queue)->prio))
        {                           /* just tack on to the end */
            insqueue((struct Gqueue *)msg, qh);
        }
        else
        {                           /* find it's place */
            while ((queue != qh) && (msg_prio > ((MsgQueue *) queue)->prio))
                queue = queue->q_tail;

            /* modified insqueue() */
            qp = queue->q_head;
            ql->q_head = qp;
            ql->q_tail = queue;
            queue->q_head = ql;
            qp->q_tail = ql;
        }
    }

    /* check for thread waiting on queue */
    if (!queueIsEmpty(&qcb->tQueue))
    {
        register Thread *td;
        td = (Thread *)qcb->tQueue.q_head;
        remqueue((struct Gqueue *)td);
        _Add_ready(td);
    }

    /* increment current message count */
    ++qcb->attr.mq_curmsgs;

    /* possible context switch */
    __pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);    //set async type
    _Relinquish(ps);
    __pthread_setcanceltype(oldtype, NULL);    //restore old type

    return 0;
}

/**
 *  @brief     Send a message to a message queue.
 *
 *  @param[in] mqdes: message queue descriptor.
 *  @param[in] msg_ptr: pointer to the send message.
 *  @param[in] msg_len: length of the message pointed to by msg_ptr.
 *  @param[in] msg_prio: priority of the selected message.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int mq_send(mqd_t mqdes, const char *msg_ptr, size_t msg_len,
         unsigned int msg_prio)
{
    return mq_timedsend(mqdes, msg_ptr, msg_len, msg_prio, NULL);
}

/**
 * @} end queue group
 */


#if 0

/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


/**
 *  @brief     Output debug information
 *
 *  @param[in] mqdes: specifies the message queue descriptor.
 *
 *  @return    none
 */
void dumpQueue(mqd_t mqdes)
{
    QCB *qcb = qd2qcb(mqdes);
    register struct Gqueue *qh;
    *qh = &qcb->mQueue, *q;

    if (queueIsEmpty(qh))
        xprintf("\tEMPTY\n");
    else
    {
        xputchar('\t');
        for (q = qh->q_head; q != qh; q = q->q_head)
            xprintf (" %d", ((MsgQueue *)q)->prio);
        xputchar('\n');
    }
}

/**
 * @} end private kernel functions group
 */

#endif

#endif /* _POSIX_MESSAGE_PASSING */
