/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"


/**
 * @addtogroup rendezvous_group rendezvous
 * @{
 */

/**
 *  @brief     Unblock the thread from which a message has been received.
 *
 *  @param[in] rply: pointer to area transmitting data.
 *  @param[in] id: thread identifier.
 *
 *  @retval    pthread_t: identifier of the thread being replied to.
 *  @retval    0: call failed. Variable errno is set appropriately. (See errno.h)
 */
pthread_t mr_reply(void *rply, pthread_t id)
{
    register Thread *ap;
    register Thread *replyee;
    register unsigned int len;
    Status ps;
    char *srplymsg;

    ap = _Active;

    ps = i_disable();
    replyee = ConvertToTD(id);

    /* If the user replies to a thread which exists but is not blocked on his
     thread, another processor may be in the process of changing the state of
     the prospective replyee at the time we are testing this state. This is
     not good. */
    if ((replyee == NULL) ||
      (replyee->td_state != REPLY_BLOCKED) ||
      (replyee->td_correspondent != _Active->td_id))
    {
        errno = EBADID;
        i_restore(ps);
        return 0;
    }

    // Log this kernel function
    DLOG_MR_REPLY_INSERT;

    /* Copy the reply message. */
    srplymsg = (char *) mp_ext2int(replyee->td_rply);
    len = MSG_SIZE (rply);
    if (len > MSG_SIZE (srplymsg))
        len = MSG_SIZE(srplymsg);
    /* This works even for TMS320Cxx */
    if (len < sizeof(short))
        len = (unsigned int)sizeof(short);

    ap->td_correspondent = id;

    /* restore ints during copy */
    i_restore(ps);
    (void)safe_copy (srplymsg, rply, len);
    ps = i_disable();

#ifdef SAM_q
    if (ap->td_correspondent == NULL)
    {
        i_restore(ps);
        errno = EBADID;
        return(0);
    }
#endif

    /*
     * Remove replyee from my reply queue.
     */
    remqueue((struct Gqueue *)replyee);

    _Add_ready(replyee);
    _Relinquish(ps);           /* In case replyee has priority */

    return (ap->td_correspondent);
}

/**
 * @} end rendezvous group
 */
