/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"


/**
 * @addtogroup rendezvous_group rendezvous
 * @{
 */

/**
 *  @brief     Send a signal to a thread
 *
 *  @param[in] tid: thread identifier represent a thread in the system.
 *  @param[in] signal_num: specfies the signal number to send taking values from zero to 31.
 *
 *  @retval    0: success.
 *  @retval    EINVAL: the calling thread has attempted to use an invalid signal number.
 *  @retval    EBADID: the calling thread has attempted to signal a tid which does not represent a thread in the system.
 */
int mr_sigpost(pthread_t id, int signum)
{
    register Thread *td;
    Status ps;

    if ((signum < 0) || (signum >= MR_SIG_MAX))
        return EINVAL;

    td = ConvertToTD (id);
    if (td == NULL)
        return EBADID;

    // Log this kernel function
    DLOG_MR_SIGPOST_INSERT;

    ps = i_disable();

    td->td_sigpend |= ((BitMask)1 << (BitMask)signum);
    if ((td->td_sigpend & td->td_sigmask) != 0U)
    {
        /* receiver is already waiting, or is sitting in sleep/nanosleep. */
        if ((td->td_state == RCV_BLOCKED) || (td->td_state == TIMER_BLOCKED))
        {
            if (td->td_state == TIMER_BLOCKED)
            {
                remqueue ((struct Gqueue *)&td->td_timeout); //remove td_timeout from timer queue
                td->td_timeout.state &= ~TIMER_ARMED;
            }
            _Add_ready(td);
            _Relinquish(ps);
            return 0;
        }
    }

    i_restore(ps);
    return 0;
}

/**
 * @} end rendezvous group
 */
