/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_time.h"
#include "_pthread.h"


#if _POSIX_TIMERS > 0

/**
 * @addtogroup time_group time
 * @{
 */

/**
 *  @brief     Suspends the current thread from execution until either the time interval has elapsed.
 *
 *  @param[in]  rqtp: pointer to suspension time.
 *  @param[out] rmtp: contains the amount of time remaining in the interval.
 *
 *  @retval    0: success.
 *  @retval    -1: error. Variable errno is set appropriately. (See errno.h)
 */
int nanosleep(const struct timespec *rqtp, struct timespec *rmtp)
{
    struct itimerspec sleeper;
    Status ps;
    int oldtype;
    int ret;

    /* Check for cancellation */
    CANCELLATION_POINT();

    if ((rqtp->tv_nsec < 0) || (rqtp->tv_nsec >= NANOSECOND))
    {
        errno = EINVAL;
        return -1;
    }

    // Log this kernel function
    DLOG_NANOSLEEP_INSERT;

    // If timeout 0, do not task switching
    if ((rqtp->tv_sec == 0) && (rqtp->tv_nsec ==0))
        return 0;

    //set async type
    __pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);

    /* Set up our built-in timer */
    (void)memset(&sleeper, 0, sizeof(struct itimerspec));
    //sleeper.it_value = *rmtp;   makes no sense - need assignment to sleep

    sleeper.it_value.tv_sec = rqtp->tv_sec;
    sleeper.it_value.tv_nsec = rqtp->tv_nsec;

    ps = i_disable();
    __timer_settime((TMCB *)(&_Active->td_timeout), 0, &sleeper, NULL);

    /* Go to sleep. */
    remqueue ((struct Gqueue *) _Active);
    _Active->td_state = TIMER_BLOCKED;

    _Relinquish(ps);

    ps = i_disable();
    /*
     * It thread's timer is armed - the nanosleep() function was terminated
     */
    if (_Active->td_timeout.state & TIMER_ARMED)
    {
        /* Fill remain time */
        if (rmtp != NULL)
        {
            rmtp->tv_sec = _Active->td_timeout.timeout.tv_sec - CurrentTime.tv_sec;
            rmtp->tv_nsec = _Active->td_timeout.timeout.tv_nsec - CurrentTime.tv_nsec;
            if (rmtp->tv_nsec < 0)
            {
                rmtp->tv_nsec += NANOSECOND;
                rmtp->tv_sec--;
            }
        }

        /* Remove the armed timer */
        remqueue ((struct Gqueue *) &_Active->td_timeout);
        _Active->td_timeout.state &= ~TIMER_ARMED;

        errno = EINTR;
        ret = -1;
    }
    else
        ret = 0;

    i_restore(ps);

    //restore old type
    __pthread_setcanceltype(oldtype, NULL);

    return ret;
}


/**
 *  @brief     Suspends the current thread from execution until either the time interval has elapsed.
 *
 *  @param[in] seconds: suspension time in seconds.
 *
 *  @return    0 if success (requested time has elapsed), otherwise - "unslept" amount  in seconds.
 */
unsigned int sleep(unsigned int seconds)
{
    struct timespec rqtp;
    struct timespec rmtp;
    int ret;
    unsigned int remaining = 0;

    rqtp.tv_sec = (time_t)seconds;
    rqtp.tv_nsec = 0;
    ret = nanosleep(&rqtp, &rmtp);

    /*
     * nanosleep() returns -1 if it was interrupted
     */
    if (ret < 0)
    {
        remaining = (unsigned int)rmtp.tv_sec;
        if ((remaining < seconds) && (rmtp.tv_nsec >= 500000000))
        {
            /* Round up */
            remaining++;
        }
    }

    return remaining;
}


/**
 *  @brief     Suspends the current thread from execution until either the time interval has elapsed.
 *
 *  @param[in] useconds: suspension time in microseconds and not less than one second.
 *
 *  @retval    0: success.
 *  @retval    -1: error. Variable errno is set appropriately. (See errno.h)
 */
int usleep(useconds_t useconds)
{
    struct timespec rqtp;

    rqtp.tv_sec = 0;
    rqtp.tv_nsec = (long)useconds * 1000L;
    return nanosleep(&rqtp, NULL);
}


/**
 * @} end time group
 */

#endif /* _POSIX_TIMERS */
