/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"

/* The following adds more debugging in rn_retseg */
#define HEAP_BOUNDS        0
#define DEBUGON            0
#define VERBOSE            0


/* header block - this must be x4 bytes */
typedef struct header {
    unsigned int magic;
#define MAGIC_MALLOC    (unsigned int)0x12121212
#define MAGIC_FREE      (unsigned int)0x23232323
    unsigned int size;
    struct header *next;
} header, *headerp;

/* Start & End Bytes */
#define START_BYTES     (sizeof(header))
#define END_BYTES       (sizeof(unsigned))
/* x is a total number of bytes available, like from get_mem */
#define USABLE_BYTES(x) ((x) - (int)(START_BYTES + END_BYTES))
/* s is the number of bytes the user requested */
#define TOTAL_BYTES(s)  ((s) + START_BYTES + END_BYTES)
#define END_OFFSET(s)   ((s) + START_BYTES)
#define END_MAGIC(h)    *(unsigned *)((char *)(h) + START_BYTES + (h)->size)
#define RET_ADDR(ptr)   ((void *)((char *)(ptr) + START_BYTES))

#define check_magic(hdr, m) ( (((hdr)->magic == (m)) && (END_MAGIC(hdr) == (m))) )

#define BAD_POOLNO(poolno)  ( ((poolno) < 0) || ((poolno) >= MAXPOOLS) )

/* pool header */
typedef struct {
    int inuse;
    header freelist;
    headerp lfree;
#if HEAP_BOUNDS
    char *start;
    char *end;
#endif
} PoolHeader;

static PoolHeader Pool[MAXPOOLS];


 /**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


 /**
 *  @brief     Set the memory with header and debug info
 *
 *  @param[in] ptr: pointer to start address of memory.
 *  @param[in] size: memory size.
 *  @param[in] flag: specific flag for memory (MAGIC_MALLOC or MAGIC_FREE).
 *
 *  @return    no return value.
 */
static void set_mem(char *ptr, unsigned int size, unsigned int flag)
{
    headerp hptr;
    unsigned int *eptr;

    /* given the pointer to the start of mem, assign the magic numbers
    and the size. */
    hptr = (headerp)ptr;
    hptr->magic = flag;
    hptr->size = size;
    eptr = (unsigned *)((char *)ptr + END_OFFSET(size));
    *eptr = flag;
}


/**
 * @} end private kernel functions group
 */


/**
 * @addtogroup malloc_group malloc
 * @{
 */


/**
 * @brief     Create memory pool.
 *
 * @param[in] poolno: handle used to access the pool.
 * @param[in] startaddr: start address of the pool area.
 * @param[in] length: length of the pool area.
 *
 * @retval    0: success.
 * @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int pool_create(int poolno, unsigned long startaddr, int length)
{
    static int PoolsInitialized = 0;
    Status ps;
    unsigned long align_startaddr;
    int len;

    // Log this kernel function
    DLOG_POOL_CREATE_INSERT;

    ps = i_disable();

    if (PoolsInitialized == 0 )
    {
        (void)memset(Pool, 0, sizeof(Pool));
        PoolsInitialized = 1;
    }

    if (BAD_POOLNO(poolno) || (Pool[(poolno)].inuse == 1))
    {
        i_restore(ps);
        errno = EINVAL;
        return -1;
    }

    /* align region on 32-bit boundry */
/******************* NEEDS ATTENTION WAS SCREWING US UP*************/
    align_startaddr = ALIGN_LONG(startaddr);
/*******************************************************************/
    /* get the actual size */
    len = length - (length % 4);    // length must be divisible by 4
    len = USABLE_BYTES(len);

    if (len <= 0)
    {
        i_restore(ps);
        errno = EINVAL;
        return -1;
    }

    /* set the memory, i.e. give the block structure  */
    set_mem((char *)align_startaddr, (unsigned int)len, MAGIC_FREE);

    /* initialize the region header */
    Pool[poolno].inuse          = 1;
    Pool[poolno].freelist.magic = 0;
    Pool[poolno].freelist.size  = 0;
    Pool[poolno].freelist.next  = (headerp)align_startaddr;
    Pool[poolno].lfree          = &Pool[poolno].freelist;
    ((headerp)align_startaddr)->next = &Pool[poolno].freelist;
#if HEAP_BOUNDS
    Pool[poolno].start          = NULL;
    Pool[poolno].end            = NULL;
#endif

    i_restore(ps);
    return 0;
}


/**
 * @brief     Destroy memory pool.
 *
 * @param[in] poolno: handle used to access the pool.
 *
 * @retval    0: success.
 * @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int pool_destroy(int poolno)
{
    if (BAD_POOLNO(poolno) || (Pool[(poolno)].inuse == 0))
        RETURN_ERRNO(EINVAL);

    // Log this kernel function
    DLOG_POOL_DESTROY_INSERT;

    Pool[poolno].inuse = 0;
    return 0;
}


/**
 * @brief     Get pointer to a dynamically allocated block.
 *
 * @param[in] poolno: handle used to access the pool.
 * @param[in] size: size in bytes of memory pool.
 *
 * @retval    0: failure.
 * @retval   (headerp): pointer to a dynamically allocated block of size bytes from memory pool.
 */
void *pool_alloc(int poolno, size_t size)
{
    headerp hdr_free, hdr_prev;
    Status ps;
    size_t align_size;

    if (size == 0U)
        return NULL;

    if (BAD_POOLNO(poolno) || (Pool[(poolno)].inuse == 0))
        return NULL;

    // Log this kernel function
    DLOG_POOL_ALLOC_INSERT;

    /* must be a x4 */
    /******** NEEDS ATTENTION ***********/
    align_size = ALIGN_LONG(size);

    /* search for a memory block big enough */
    ps = i_disable();
    hdr_prev= Pool[poolno].lfree;

    for (hdr_free = hdr_prev->next; ; hdr_free = hdr_free->next)
    {
#if DEBUGON
        if ((free != &(Pool[poolno].freelist)) && (check_magic(hdr_free, MAGIC_FREE) == 0))
        {
            xprintf("rn_getseg: BAD MAGIC NUMBER! CORRUPTED HEAP!\n");
            return 0;
        }
#endif

        /* free->size is a count of the USABLE BYTES.
         * looking for first memory equal to or bigger.
         */
        if ((hdr_free->size == align_size) || (hdr_free->size >= TOTAL_BYTES(size)))
        {
            if (hdr_free->size == align_size)
            {
                /* exact match - just delink from free list */
                hdr_prev->next = hdr_free->next;
                /* set memory as alloced */
                set_mem((char *)hdr_free, hdr_free->size, MAGIC_MALLOC);
            }
            else
            {
                /* reduce size of current block so we can strip off the end. */
                hdr_free->size -= (unsigned int)TOTAL_BYTES(align_size);
                /* set the memory of the reduced free block */
                set_mem((char *)hdr_free, hdr_free->size, MAGIC_FREE);
                hdr_free = (headerp)((char *)hdr_free + TOTAL_BYTES(hdr_free->size));
                /* set the memory of the tail portion. */
                set_mem((char *)hdr_free, (unsigned int)align_size, MAGIC_MALLOC);
            }

            /* set previous ptr to the place before the just-malloced
             * memory so that if that memory gets freed right away,
             * we will be ready at that spot.
             */
            Pool[poolno].lfree = hdr_prev;

#if HEAP_BOUNDS
            if ((unsigned)hdr_free < (unsigned)Pool[poolno].start)
                Pool[poolno].start = (char *)hdr_free;

            if ((unsigned)hdr_free > (unsigned)Pool[poolno].end)
                Pool[poolno].end = (char *)hdr_free;
#endif

            i_restore(ps);

            return RET_ADDR(hdr_free);
        }

        if (hdr_free == Pool[poolno].lfree)
            /* Traversed entire free list. */
            break;

        hdr_prev = hdr_free;
    }

    errno = ENOMEM;
    i_restore(ps);
    return NULL;
}


/**
 * @brief     Free allocated memory block.
 *
 * @param[in] poolno: handle used to access the pool.
 * @param[in] buf: pointer to free memory.
 *
 * @retval    0: success.
 * @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int pool_free(int poolno, void *buf)
{
    headerp hdr, hdr_free;
    Status ps;

    if (BAD_POOLNO(poolno) || (Pool[(poolno)].inuse == 0))
        RETURN_ERRNO(EINVAL);

    if (buf == NULL)
        RETURN_ERRNO(EINVAL);

    hdr = (headerp)((char *)buf - START_BYTES);

    if (((uint_32)hdr & (uint_32)(sizeof(long) - 1U)) != (uint_32)0)
    {
#if VERBOSE
        xprintf("u_pool_free: misaligned buf %x\n", buf);
#endif
        RETURN_ERRNO(EMSGSIZE);
    }

#if HEAP_BOUNDS
    if ((char *)hdr < Pool[poolno].start || (char *)hdr > Pool[poolno].end)
    {
#if VERBOSE
        xprintf("u_pool_free: address %x not in Pool[poolno]\n", buf);
#endif
        RETURN_ERRNO(ENOENT);
    }
#endif

    /*check to make sure the pointer passed in has a MAGIC_MALLOC */
    if (hdr->magic != MAGIC_MALLOC)
    {
#if VERBOSE
        if (hdr->magic == MAGIC_FREE)
            xprintf("u_pool_free: address %x already freed\n", buf);
        else
            xprintf("u_pool_free: address %x bad magic\n", buf);
#endif
        RETURN_ERRNO(EAGAIN);
    }

    if (END_MAGIC(hdr) != MAGIC_MALLOC)
    {
#if VERBOSE
        xprintf("u_pool_free: address %x end magic overwritten\n", buf);
#endif
        RETURN_ERRNO(E2BIG);
    }

    // Log this kernel function
    DLOG_POOL_FREE_INSERT;

    /* Find where this block fits in the list.
     * Should be in between two elements in the list.
     */
    ps = i_disable();

    for (hdr_free = Pool[poolno].lfree; !((hdr > hdr_free) && (hdr < hdr_free->next)); hdr_free = hdr_free->next)
        /* if free is greater then its next pointer, we are at the end
         * of the list. AND block header must be either greater then the end
         * or before the beginning.
         */
        if ((hdr_free >= hdr_free->next) && ((hdr > hdr_free) || (hdr < hdr_free->next)))
            break;

    /* if end of this block equals the start of the next, connect to upper */
    if (((char *)hdr + TOTAL_BYTES(hdr->size)) == (char *)hdr_free->next)
    {
        hdr->size += (unsigned int)TOTAL_BYTES(hdr_free->next->size);
        hdr->next = hdr_free->next->next;
        set_mem((char *)hdr, hdr->size, MAGIC_FREE);
    }
    else
    {
        set_mem((char *)hdr, hdr->size, MAGIC_FREE);
        hdr->next = hdr_free->next;
    }

    /* if end of previous block equals this block, connect to lower */
    if (((char *)hdr_free + TOTAL_BYTES(hdr_free->size)) == (char *) hdr)
    {
        hdr_free->size += (unsigned int)TOTAL_BYTES(hdr->size);
        hdr_free->next = hdr->next;
        set_mem((char *)hdr_free, hdr_free->size, MAGIC_FREE);
    }
    else
        hdr_free->next = hdr;

    Pool[poolno].lfree = hdr_free;
    i_restore(ps);
    return 0;
}


/**
 * @} end memory group
 */
