/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_pthread.h"


/**
 * @addtogroup pthread_group pthread
 * @{
 */


/**
 *  @brief Set cancelability state
 *
 *  @param [in] state: set thread's cancelability state.
 *  @param [in] oldstate: pointer to a variable, for get thread's cancelability oldstate.
 *
 *  @retval     EINVAL: error. not valid parameter.
 *  @retval     0: success
 */
int pthread_setcancelstate(int state, int *oldstate)
{
    Status ps;
    unsigned int oldval, newval;

    if ((state != PTHREAD_CANCEL_ENABLE) && (state != PTHREAD_CANCEL_DISABLE))
        return EINVAL;

    // Log this kernel function
    DLOG_PTHREAD_SETCANCELSTATE_INSERT;

    ps = i_disable();
    if (oldstate != NULL)
        *oldstate = (((_Active->cancelstatus & CANCELSTATE_BITMASK) != 0U)
                ? PTHREAD_CANCEL_DISABLE : PTHREAD_CANCEL_ENABLE);

    oldval = _Active->cancelstatus;
    newval = ((state == PTHREAD_CANCEL_DISABLE)
            ? (oldval | CANCELSTATE_BITMASK)
            : (oldval & ~CANCELSTATE_BITMASK));
    _Active->cancelstatus = newval;

    if (CANCEL_ENABLED_AND_CANCELING_AND_ASYNCHRONOUS(newval))
    {
        _kexit(PTHREAD_CANCELED);
    }

    i_restore(ps);
    return 0;
}


/**
 * @} end pthread group
 */
