/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "kernel.h"
#include "directory.h"
#include "threads_utilization.h"
#include "idle.h"
#include "dirstubs.h"


int _threads_count = 0;
thread_util_t *_threads_table;


/**
 * @addtogroup misc_group misc
 * @{
 */

/**
 *  @brief     Gets the current number of threads in the system.
 *
 *
 *  @return    number of registered threads and servers.
 */
int threads_numb(void)
{
    register TabEntry *tab;
    int entries_cnt = 0;    //counter of entries
    Status ps;

    ps = i_disable();
    for (tab = tHead; tab != NULL; tab = tab->tab_next)
    {
        if ((tab->tab_type == TYPE_THREAD) || (tab->tab_type == TYPE_SERVER))
        {
            entries_cnt++;
        }
    }
    i_restore(ps);
    return entries_cnt + 1;    // +1 for idle task
}

/**
 *  @brief     Starts the utilization measurement.
 *
 *  @param[in] threads_table: structures allocated for the thread table contain the following information for each thread.
 *  @param[in] threads_num: number of threads.
 *
 *  @return    none.
 */
void threads_utilization_start(thread_util_t *threads_table, int threads_num)
{
    register TabEntry *tab;
    int i = 0;
    Status ps;

    // IdleThread
    threads_table[0].thread_id = _IdleID;
    threads_table[0].thread_name = _IdleName;
    threads_table[0].util = 0;

    ps = i_disable();
    for (tab = tHead; tab != NULL; tab = tab->tab_next)
    {
        if ((tab->tab_type == TYPE_THREAD) || (tab->tab_type == TYPE_SERVER))
        {
            i++;
            threads_table[i].thread_id = tab->tab_value;
            threads_table[i].thread_name = tab->tab_name;
            threads_table[i].util = 0;
        }
    }
    //Start utilization count
    _threads_table = threads_table;
    _threads_count = threads_num;
    i_restore(ps);
}


/**
 *  @brief     Indicates the end of the utilization measurements.
 *
 *  @return    none
 */
void threads_utilization_stop(void)
{
    Status ps;
    u_long ticks = 0;
    int tmp, i;

    tmp = _threads_count;

    //Stop utilization count
    ps = i_disable();
    _threads_count = 0;
    i_restore(ps);

    // count ticks
    for (i = 0; i < tmp; i++)
    {
        ticks += _threads_table[i].util;
    }

    // percent
    for (i = 0; i < tmp; i++)
    {
        u_long whole;
        u_long fractional;
        whole = (_threads_table[i].util * 100UL) / ticks;
        fractional = ((_threads_table[i].util * 100UL) % ticks) * 10UL / ticks;

        _threads_table[i].util = ((u_long)whole << 16) | fractional;
    }

}

/**
 * @} end misc group
 */
