/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <stdarg.h>             /* provided by compiler */
#include <math.h>


#define HUGEPREC    30000
#define FLOATS      0


#if FLOATS
    #define CVTBUFSIZE  320    /* assert(CVTBUFSIZE > log10(MAXFLOAT)) */
    #define ABUFSIZE    (CVTBUFSIZE + 20)
    static char *ecvtr(double number, int ndigits, int *decpt, int *sign, char *buf);
    static char *fcvtr(double number, int ndigits, int *decpt, int *sign, char *buf);
    static char *gcvtr(double number, int ndigit, char *buf, char *wbuf);
#else
    #define ABUFSIZE    20
#endif


/**
 * @addtogroup misc_group misc
 * @{
 */

/**
 *  @brief     Print a string through xputchar.
 *
 *  @param[in] s: pointer to the string.
 *
 *  @retval    none.
 */
void xputs(const char *s)
{
    Status ps;
    const char *str = s;

    ps = i_disable();
    while (*str != '\0')
    {
        xputchar((int)*str);
        str++;
    }
    xputchar((int)'\n');
    i_restore(ps);
}


/**
 *  @brief     Create generic polled formatted output.
 *
 *  @param[in] format: character string which contains two types of objects.
 *  @param[in] args: list of parameters.
 *
 *  @return    Always return 1.
 */
int xprintf(const char *format, ...)
{
    va_list args;
    const char *formatp = format;

    register char   *sp;
    int             fwidth, prec, ljustify, longarg, len, sign;
    char            padchar;
    unsigned long   ul;
#if FLOATS
    char            *dp;
    int             decpt;
    double          dbl;
    char            fbuf[CVTBUFSIZE];
#endif
    char        abuf[ABUFSIZE];
    static char hextab[] = "0123456789ABCDEF";
    Status ps;


    ps = i_disable();

    va_start (args, format);

    while (*formatp != '\0')
    {

        if (*formatp != '%')
        {
            xputchar((int)*formatp++);
            continue;
        }

        formatp++;

        ljustify = (*formatp == '-') ? 1 : 0;
        if ( ljustify != 0) formatp++;

        padchar = ' ';
        fwidth = 0;
        if ((*formatp >= '0') && (*formatp <= '9'))
        {
            if (*formatp == '0')
                padchar = '0';
            do
                fwidth = (10 * fwidth) + (*formatp++) - '0';
            while ((*formatp >= '0') && (*formatp <= '9'));
        }
        else if (*formatp == '*')
        {
            formatp++;
            fwidth = va_arg(args, int);
        }

        prec = HUGEPREC;         /* Very big. */
        if (*formatp == '.')
        {
            formatp++;
            if ((*formatp >= '0') && (*formatp <= '9'))
            {
                prec = 0;
                do
                    prec = (10*prec) + (*formatp++) - '0';
                while ((*formatp >= '0') && (*formatp <= '9'));
            }
            else if (*formatp == '*')
            {
                formatp++;
                prec = va_arg(args, int);
            }
        }

        longarg = (*formatp == 'l') ? 1 : 0;
        if ( longarg != 0) formatp++;

        sp = &abuf[ABUFSIZE];

        switch (*formatp++)
        {
            case 'd':
            case 'i':
                ul = (longarg != 0) ? va_arg(args, long) : va_arg(args, int);
                sign = ((long)ul < 0) ? 1 : 0;
                if (sign != 0) ul = (unsigned long) (-(long)ul);
                do
                {
                    *--sp = hextab[ul % 10UL];
                    ul /= 10UL;
                } while (ul != 0UL);
                if (sign != 0)
                    *--sp = '-';
                goto numeric;
                break;    // for MISRA compliance
            case 'u':
                ul = (longarg != 0) ? va_arg(args, unsigned long) :
                    va_arg(args, unsigned);
                do
                {
                    *--sp = hextab[ul % 10UL];
                    ul /= 10UL;
                } while (ul != 0UL);
                goto numeric;
                break;    // for MISRA compliance
            case 'o':
                ul = (longarg != 0) ? va_arg(args, unsigned long) :
                    va_arg(args, unsigned);
                do
                {
                    *--sp = hextab[ul & 0x7UL];
                    ul >>= 3;
                } while (ul != 0UL);
                goto numeric;
                break;    // for MISRA compliance
            case 'x':
            case 'h':     /* Added for compatibility with Harmony */
                ul = (longarg != 0) ? va_arg(args, unsigned long) :
                    va_arg(args, unsigned);
                do
                {
                    *--sp = hextab[ul & 0xfUL];
                    ul >>= 4;
                } while (ul != 0UL);
                goto numeric;
                break;    // for MISRA compliance
            case 'p':
                padchar = '0';
                fwidth = (int)sizeof(int) * 2;
                ul = va_arg(args, unsigned int);
                do
                {
                    *--sp = hextab[ul & 0xfUL];
                    ul >>= 4;
                } while (ul != 0UL);
numeric:
                len = (int)&abuf[ABUFSIZE] - (int)sp;
                if (prec != HUGEPREC) {
                    while ((len < prec) && (len < ABUFSIZE)) {
                        *--sp = '0';
                        len++;
                    }
                }
                if (padchar == '0')
                    ljustify = 0;
                break;
            case 's':
                sp = va_arg(args, char *);
                if (sp == NULL) sp = "(null)";
                len = (int)strlen(sp);
                if (len > prec)
                    len = prec;
                padchar = (int)' ';
                break;
            case 'c':        /* Chars are passed as int's */
                *--sp = va_arg(args, int);
                len = 1;
                padchar = ' ';
                break;
#if FLOATS
            case 'f':
                dbl = va_arg(args, double);
                if (isnan(dbl) || isinf(dbl)) {
                    sp = isnan(dbl) ? "NaN" : "Infinity";
                    len = (int)strlen(sp);
                    padchar = ' ';
                    break;
                }
                if (prec == HUGEPREC)
                    prec = 6;
                dp = fcvtr(dbl, prec, &decpt, &sign, fbuf);
                sp = abuf;
                if (sign != 0)
                    *sp++ = '-';
                if (decpt <= 0) {
                    *sp++ = '0';
                    *sp++ = '.';
                    while ((decpt != 0) && (prec != 0)) {
                        decpt++;
                        prec--;
                        *sp++ = '0';
                    }
                }
                else {
                    while ((decpt-- != 0) && (*dp != '\0'))
                        *sp++ = *dp++;
                    if (*dp != '\0')
                        *sp++ = '.';
                }
                while (*dp != '\0')
                    *sp++ = *dp++;
                goto floating;
                break;    // for MISRA compliance
            case 'e':
                dbl = va_arg(args, double);
                if (isnan(dbl) || isinf(dbl))
                {
                    sp = isnan(dbl) ? "NaN" : "Infinity";
                    len = (int)strlen(sp);
                    padchar = ' ';
                    break;
                }
                prec = (prec == HUGEPREC) ? 7 : (prec + 1);
                dp = ecvtr(dbl, prec, &decpt, &sign, fbuf);
                sp = abuf;
                if (sign != 0)
                    *sp++ = '-';
                if (*dp == '0')    /* Special case for zero so don't */
                    decpt++;/* get an exponent of -1      */
                *sp++ = *dp++;
                *sp++ = '.';
                while (*dp != '\0')
                    *sp++ = *dp++;
                *sp++ = 'e';
                decpt -= 1;
                if (decpt < 0)
                {
                    *sp++ = '-';
                    decpt = -decpt;
                }
                else
                    *sp++ = '+';

                *sp++ = (decpt/100) + '0';
                *sp++ = ((decpt % 100)/10) + '0';
                *sp++ = (decpt%10) + '0';
                goto floating;
                break;    // for MISRA compliance
            case 'g':
                dbl = va_arg(args, double);
                if (isnan(dbl) || isinf(dbl))
                {
                    sp = isnan(dbl) ? "NaN" : "Infinity";
                    len = (int)strlen(sp);
                    padchar = ' ';
                    break;
                }
                if (prec == HUGEPREC)
                    prec = 6;
                sp = gcvtr(dbl, prec, abuf, fbuf);
                while (*sp != '\0')
                    sp++;
floating:
                len = (int)sp - (int)abuf;
                sp = abuf;
                break;
#endif
            default:
                *--sp = format[-1];
                len = 1;
                padchar = ' ';
                break;
        }

        if (ljustify == 0)
            while (fwidth-- > len)
                xputchar((int)padchar);

        for (ul = (unsigned long)len; ul-- != 0UL; )
            xputchar((int)*sp++);

        if (ljustify != 0)
            while (fwidth-- > len)
                xputchar((int)padchar);
    }

    va_end(args);

    i_restore(ps);
    return 1;        //dummy value
}


#if FLOATS


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */

/**
 *  @brief     Convert a floating-point number to a string.
 *
 *  @param[in] value: variable value.
 *  @param[in] ndigits: number of digits after the decimal point.
 *  @param[in] decpt: pointer to the start of the string.
 *  @param[in] sign: sign of variable value.
 *  @param[in] eflag: flag determines the style.
 *  @param[out]buf: buffer allocated for the result.
 *
 *  @retval    Pointer to the string.
 */
static char* cvt(double value, int ndigits, int *decpt, int *sign, int eflag,char *buf)
{
    register int r2;
    double fi, fj;
    register char *p, *p1;
    int digits = ndigits;
    double arg = value;

    if (digits < 0)
        digits = 0;
    if (digits >= (CVTBUFSIZE - 1))
        digits = CVTBUFSIZE - 2;
    r2 = 0;
    *sign = 0;
    p = &buf[0];
    if (arg < 0.0)
    {
        *sign = 1;
        arg = -arg;
    }
    arg = modf(arg, &fi);
    p1 = &buf[CVTBUFSIZE];
    /*
     * Do integer part
     */
/*
 * [++doug] May 4/90. "oldway" doesn't work on the 29000 at least using the
 * High C compiler. If "fi" is 10.0, it appears that "fi/10" is just slightly
 * less than 1.0 (ie. 0.9999...). Thus, after calling "modf", "fi" will be
 * zero, and "fj" will be 0.9999... When "fj" is then rounded up, we get
 * 10, and 10 + '0' is ':'. My fix is to round up "fi/10" before calling
 * modf. Don't think this breaks anything, but I'm not a numerical analyst
 * so I can't be sure.
 * I also don't know if the above behaviour is a bug on the part of either
 * the High C compiler, or the runtime libaries, or what. It's possible that
 * it's not a bug at all, and that I was relying on a non-portable assumption
 * before.
 */
    if (fi != 0.0)
    {
        p1 = &buf[CVTBUFSIZE];
        while (fi != 0.0)
        {
#ifdef oldway
            fj = modf(fi / 10.0, &fi);
            fj = (fj + 0.03) * 10.0;
            *--p1 = (int)fj + '0';
#else
            fj = modf((fi / 10.0) + 0.03, &fi);
            fj = fj * 10.0;
            *--p1 = (int)fj + '0';
#endif
            r2++;
        }
        while (p1 < &buf[CVTBUFSIZE])
            *p++ = *p1++;
    }
    else if (arg > 0.0)
    {
        fj = arg * 10.0;
        while (fj < 1.0)
        {
            arg = fj;
            r2--;
            fj = arg * 10.0;
        }
    }
    p1 = &buf[digits];
    if (eflag == 0)
        p1 += r2;
    *decpt = r2;
    if (p1 < &buf[0])
    {
        buf[0] = '\0';
        return(buf);
    }

    while ((p <= p1) && (p < &buf[CVTBUFSIZE]))
    {
        arg *= 10.0;
        arg = modf(arg, &fj);
        *p++ = (int)fj + '0';
    }

    if (p1 >= &buf[CVTBUFSIZE])
    {
        buf[CVTBUFSIZE-1] = '\0';
        return (buf);
    }

    p = p1;
    *p1 += 5;
    while (*p1 > '9')
    {
        *p1 = '0';
        if (p1 > buf)
            ++*--p1;
        else
        {
            *p1 = '1';
            (*decpt)++;
            if (eflag == 0)
            {
                if (p > buf)
                    *p = '0';
                p++;
            }
        }
    }
    *p = '\0';
    return(buf);
}

/**
 *  @brief     Converts the exponent variable to a null-terminated string.
 *
 *  @param[in] number: variable value.
 *  @param[in] ndigits: number of digits after the decimal point.
 *  @param[in] decpt: pointer to the start of the string.
 *  @param[in] sign: sign of variable value.
 *  @param[out]buf: buffer allocated for the result.
 *
 *  @return    Pointer to the string.
 */
static char *ecvtr(double number, int ndigits, int *decpt, int *sign, char *buf)
{
    return(cvt(number, ndigits, decpt, sign, 1, buf));
}

/**
 *  @brief     Convert a floating-point number to a string.
 *
 *  @param[in] number: variable value.
 *  @param[in] ndigits: number of digits after the decimal point.
 *  @param[in] decpt: pointer to the start of the string.
 *  @param[in] sign: sign of variable value.
 *  @param[out]buf: buffer allocated for the result.
 *
 *  @retval    Pointer to the string.
 */
static char *fcvtr(double number, int ndigits, int *decpt, int *sign, char *buf)
{
    return(cvt(number, ndigits, decpt, sign, 0, buf));
}


/**
 *  @brief     Converts the variable in the style f or e to a string.
 *
 *  @param[in] number: variable value.
 *  @param[in] ndigits: number of digits after the decimal point.
 *  @param[out]buf: buffer allocated for the result.
 *  @param[in] wbuf: temporary allocated buffer for the result.
 *
 *  @retval    Pointer to the string.
 */
static char *gcvtr(double number, int ndigit, char *buf, char *wbuf)
{
    int sign, decpt;
    register char *p1, *p2;
    register int i;
    int digit = ndigit;

    p1 = ecvtr(number, digit, &decpt, &sign, wbuf);
    p2 = buf;

    if (sign != 0)
        *p2++ = '-';

    if (((decpt >= 0) && ((decpt - digit) > 0))
     || ((decpt < 0) && (decpt < - 3)))
    { /* use E-style */
        for (i = digit - 1; (i > 0) && (p1[i] == '0'); i--)
            digit--;
        decpt--;
        *p2++ = *p1++;
        if (digit > 1)
        {
            *p2++ = '.';
            for (i = 1; i < digit; i++)
                *p2++ = *p1++;
        }
        *p2++ = 'e';
        if (decpt < 0)
        {
            decpt = -decpt;
            *p2++ = '-';
        }
        else
            *p2++ = '+';
        *p2++ = (decpt / 100) + '0';
        *p2++ = ((decpt % 100) / 10) + '0';
        *p2++ = (decpt % 10) + '0';
    }
    else
    {
        if (decpt <= 0)
        {
            if (*p1 != '0')
            {
                *p2++ = '0';
                *p2++ = '.';
            }
            while (decpt < 0)
            {
                decpt++;
                *p2++ = '0';
            }
        }
        for (i = digit - 1; (i > 0) && (p1[i]=='0'); i--)
            digit--;
        for (i = 1; i <= digit; i++)
        {
            *p2++ = *p1++;
            if (i == decpt)
                *p2++ = '.';
        }
        if (digit < decpt)
        {
            while (digit++ < decpt)
                *p2++ = '0';
            *p2++ = '.';
        }
    }

    if (p2[-1] == '.')
        p2--;
    *p2 = '\0';
    return(buf);
}


/**
 * @} end private kernel functions group
 */

#endif
