/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>

#if _POSIX_MESSAGE_PASSING > 0

#include <mrendez.h>
#include "dlog_kernel_ev.h"
#include "_mqueue.h"
#include "_time.h"
#include "_pthread.h"


/**
 * @addtogroup queue_group queue
 * @{
 */


/**
 *  @brief     Receive a message from a message queue.
 *
 *  @param[in]  mqdes: message queue descriptor.
 *  @param[out] msg_ptr: pointer to the receive message.
 *  @param[in]  msg_len: size of the buffer pointed by msg_ptr.
 *  @param[out] msg_prio: priority of the selected message.
 *  @param[in]  timeout: the time interval.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
ssize_t mq_timedreceive(mqd_t mqdes, char *msg_ptr, size_t msg_len,
                 unsigned int *msg_prio, const struct timespec *timeout)
{
    Status ps;
    QCB *qcb;
    MsgQueue *msg;
    int oldtype;
    size_t recv_len;

    /* Check for cancellation */
    CANCELLATION_POINT();

    if (!VALID_MQUEUE(mqdes))
        RETURN_ERRNO(EBADF);
    qcb = qd2qcb(mqdes);
    if (qcb == NULL)
        RETURN_ERRNO(EBADF);
    if (RECVOK(mqdes) == 0)
        RETURN_ERRNO(EBADF);
    if (msg_len < (size_t)qcb->attr.mq_msgsize)
        RETURN_ERRNO(EMSGSIZE);

    // Log this kernel function
    DLOG_MQ_RECEIVE_INSERT;

    /* check if message waiting */
    ps = i_disable();

    if (queueIsEmpty(&qcb->mQueue))
    {                             /* no messages pending */
        if ((((QD *)mqdes)->mq_flags & O_NONBLOCK) != 0)
        {
            i_restore (ps);
            RETURN_ERRNO (EAGAIN);
        }

#if _POSIX_TIMEOUTS > 0
        if (timeout != NULL)
        {                           /* put on timer queue */
            struct itimerspec itimer;


            if ((timeout->tv_nsec < 0) || (timeout->tv_nsec >= 1000000000))
            {
                i_restore(ps);
                RETURN_ERRNO(EINVAL);
            }

            itimer.it_value.tv_sec = timeout->tv_sec;
            itimer.it_value.tv_nsec = timeout->tv_nsec;
            itimer.it_interval.tv_sec = 0;
            itimer.it_interval.tv_nsec = 0;
            __timer_settime((TMCB *)(&_Active->td_timeout), 0, &itimer, NULL);
        }
#endif /* _POSIX_TIMEOUTS */

        do
        {                    /* no messages pending - block */
            _Active->td_state = MQ_BLOCKED;
            remqueue((struct Gqueue *)_Active);
            insqueue_prio((struct Gqueue *)_Active, &qcb->tQueue);
            __pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);    //set async type
            _Relinquish(ps);
            __pthread_setcanceltype(oldtype, NULL);    //restore old type
#if _POSIX_TIMEOUTS > 0
            if (_Active->td_state == TIMER_EXPIRED)
                RETURN_ERRNO(ETIMEDOUT);
#endif /* _POSIX_TIMEOUTS */
            ps = i_disable();
        }
        while (queueIsEmpty(&qcb->mQueue));

#if _POSIX_TIMEOUTS > 0
        if (timeout != NULL)
            /* disarm timer */
            __timer_settime((TMCB *)(&_Active->td_timeout), 0, NULL, NULL);
#endif /* _POSIX_TIMEOUTS */
    }

    /* messages are in order */
    msg = (MsgQueue *)qcb->mQueue.q_head;

    /* remove the message from the queue before restoring ints */
    remqueue(&msg->queue);

    i_restore(ps);

    recv_len = (size_t)msg->len;
    (void)mp_memcpy (msg_ptr, ((char *) msg) + sizeof (MsgQueue), recv_len);
    if (msg_prio != NULL)
        *msg_prio = msg->prio;

    ps = i_disable();

    /* put the message on the free queue */
    insqueue ((struct Gqueue *) msg, &qcb->fQueue);
    --qcb->attr.mq_curmsgs;

    /* check for any threads blocked waiting for a message to free up */
    if (!queueIsEmpty(&qcb->rQueue))
    {
        Thread *td = (Thread *)qcb->rQueue.q_head;
        remqueue((struct Gqueue *)td);
        _Add_ready(td);
    }

    i_restore(ps);

    return (ssize_t)recv_len;
}


/**
 *  @brief     Receive a message from a message queue.
 *
 *  @param[in]  mqdes: message queue descriptor.
 *  @param[out] msg_ptr: pointer to the receive message.
 *  @param[in]  msg_len: size of the buffer pointed by msg_ptr.
 *  @param[out] msg_prio: priority of the selected message.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
ssize_t mq_receive(mqd_t mqdes, char *msg_ptr, size_t msg_len, unsigned int *msg_prio)
{
    return mq_timedreceive(mqdes, msg_ptr, msg_len, msg_prio, NULL);
}

/**
 * @} end queue group
 */

#endif /* _POSIX_MESSAGE_PASSING */
