/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"


/**
 * @addtogroup rendezvous_group rendezvous
 * @{
 */

/**
 *  @brief     Send a message to a thread
 *
 *  @param[in] rqst: pointer to a message to be sent to the thread.
 *  @param[out]rply: pointer to a message to be reply pass back data.
 *  @param[in] id: thread identifier.
 *  @param[in] port: argument specifies the port on the receiver's end through which the message is to be routed.
 *  @param[in] timeout: pointer to structure timespec with timeout value.
 *
 *  @retval    pthread_t: positive value for is the identifier of the thread which received the message.
 *  @retval    0: indicates an error.
 */
pthread_t mr_send(void *rqst, void *rply, pthread_t id, int port,
         const struct timespec *timeout)
{
    register Thread *ap;
    register Thread *receiver;
    Status ps;

    (void)timeout;

    // Log this kernel function
    DLOG_MR_SEND_INSERT;

    ap = _Active;

    ap->td_rqst = mp_int2ext(rqst);
    ap->td_rply = mp_int2ext(rply);
    ap->td_correspondent = id;
    ap->td_sendport = port;

    ps = i_disable();
    receiver = ConvertToTD(id);

    if (receiver == NULL)
    {
        errno = EBADID;
        i_restore(ps);
        return 0;
    }

    /* Check if mr_init() was called in the receiver task. */
    if (receiver->td_rd == NULL)
    {
        errno = EIO;
        i_restore(ps);
        return 0;
    }

    if ((port < 0) || (port > receiver->td_rd->td_nports))
    {
        errno = EBADPORT;
        i_restore(ps);
        return 0;
    }

    /* Remove ourselves from our ready queue and put us at the tail of the
       receiver's send queue. */
    _Active->td_state = SEND_BLOCKED;
    movequeue((struct Gqueue *)_Active, &receiver->td_rd->td_sendq[port]);

    /* tell receiver that a message waiting */
    receiver->td_rd->td_portpend |= ((BitMask)1 << (BitMask)port);

    if ((receiver->td_rd->td_portmask & receiver->td_rd->td_portpend) != 0U)
    {                           /* receiver is waiting on port */
        if (receiver->td_state == RCV_BLOCKED)
            _Add_ready(receiver);
        else if (receiver->td_state == TIMER_BLOCKED)
        {
            remqueue((struct Gqueue *)&receiver->td_timeout); //  remove td_timeout from timer queue
            receiver->td_timeout.state &= ~TIMER_ARMED;
            _Add_ready (receiver);
        }
    }
    _Relinquish(ps);

    return ap->td_correspondent;
}

/**
 * @} end rendezvous group
 */
