/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <kalloc.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"


/* Fast allocator of fixed size blocks. A singly linked list of
 * available blocks is kept. Free's blocks are put on the beginning
 * of the list, and allocated blocks are taken off the beginning of the
 * list.
 */

/* Each block in the pool is preceeded by a PT_BLOCKHEADER */
typedef struct BlockHdr {
    struct BlockHdr *ptb_next;
} BlockHdr;

/* Each pool of blocks has a PARTITION_HEADER */
typedef struct {
    BlockHdr    *pth_next;
    void        *mem;
} PtHdr;


#define MAXPARTITIONS    8
static PtHdr pt_header[MAXPARTITIONS];


/**
 * @addtogroup malloc_group malloc
 * @{
 */


/**
 * @brief     Creates a partition.
 *
 * @param[in] blocksize: size of memory blocks.
 * @param[in] numblocks: number blocks.
 *
 * @retval   ptno: return system identifier.
 * @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int pt_create(size_t blocksize, int numblocks)
{
    PtHdr *pth;
    BlockHdr *ptb;
    int ptno, i;
    char *mem;
    size_t alloc_size;

    if ((blocksize == 0U) || (numblocks == 0))
    {
        errno = EINVAL;
        return -1;
    }

    // Log this kernel function
    DLOG_PT_CREATE_INSERT;

    /* find an empty partition */
    for (ptno = 0; ptno < MAXPARTITIONS; ++ptno)
        if (pt_header[ptno].mem == NULL)
        {
            pth = &pt_header[ptno];
            alloc_size = (blocksize + sizeof(BlockHdr) + 3U) & ~3U;
            mem = kalloc(KA_PT, alloc_size * (size_t)numblocks);

            if (mem == NULL)
            {
                errno = ENOSPC;
                return -1;
            }

            pth->mem = mem;
            i = numblocks;

            for (pth->pth_next = NULL; i != 0; --i)
            {
                ptb = (BlockHdr *)mem;
                ptb->ptb_next = pth->pth_next;
                pth->pth_next = ptb;
                mem += alloc_size;
            }

            return ptno;
        }

    errno = EAGAIN;
    return -1;
}


/**
 * @brief     Destroy a partition.
 *
 * @param[in] ptno: system identifier.
 *
 * @retval    ptno: return system identifier.
 * @retval    -1: error. Variable errno is set appropriately. (See errno.h)
 */
int pt_destroy(int ptno)
{
    if ((ptno < 0) || (ptno >= MAXPARTITIONS))
    {
        errno = EBADID;
        return -1;
    }

    if (pt_header[ptno].mem == NULL)
    {
        errno = EBUSY;
        return -1;
    }

    // Log this kernel function
    DLOG_PT_DESTROY_INSERT;

    kfree(KA_PT, pt_header[ptno].mem);
    pt_header[ptno].mem = NULL;
    return ptno;
}


/**
 * @brief     Get a fixed sized buffer from the partition.
 *
 * @param[in] ptno: system identifier.
 *
 * @retval    ptb: pointer to the buffer.
 * @retval    0: error.
 */
void *pt_getblock(int ptno)
{
    BlockHdr *ptb;
    Status ps;

    if ((ptno < 0) || (ptno >= MAXPARTITIONS))
        return NULL;

    ps = i_disable();
    ptb = pt_header[ptno].pth_next;

    if (ptb == NULL)
    {
        i_restore(ps);
        return NULL;
    }

    pt_header[ptno].pth_next = ptb->ptb_next;
    i_restore(ps);

    // Log this kernel function
    DLOG_PT_GETBLOCK_INSERT;

    return (char *)(ptb + 1);
}


/**
 * @brief     Get a fixed sized buffer pointed to the partition.
 *
 * @param[in] ptno: system identifier.
 * @param[in] block: pointed to the partition.
 *
 * @return    no return value.
 */
void pt_freeblock(int ptno, void *block)
{
    BlockHdr *ptb;
    Status ps;

    if ((ptno < 0) || (ptno >= MAXPARTITIONS))
        return;

    if (pt_header[ptno].mem == NULL)
        return;

    // Log this kernel function
    DLOG_PT_FREEBLOCK_INSERT;

    ptb = (BlockHdr *)((char *)block - sizeof(*ptb));
    ps = i_disable();
    ptb->ptb_next = pt_header[ptno].pth_next;
    pt_header[ptno].pth_next = ptb;
    i_restore(ps);
}


/**
 * @} end memory group
 */
