/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_pthread.h"

void set_exit_addr(Thread *td);


/**
 * @addtogroup pthread_group pthread
 * @{
 */


/**
 *  @brief Cancellation of pthread
 *
 *  @param [in] thread: thread that will be canceled
 *
 *  @retval    0: success.
 *  @retval    ESRCH: error. No thread could be found the given thread ID
 */

int pthread_cancel(pthread_t thread)
{
    Status ps;
    Thread *td;

    // check valid of thread
    if (thread == (pthread_t)0)
    {
        return ESRCH;
    }

    ps = i_disable();

    // find the thread
    td = ConvertToTD(thread);
    if (td == NULL)
    {
        i_restore(ps);
        return ESRCH;
    }

    // Log this kernel function
    DLOG_PTHREAD_CANCEL_INSERT;

    td->cancelstatus |= CANCELING_BITMASK;

    // async cancel?
    if (CANCEL_ENABLED_AND_CANCELING_AND_ASYNCHRONOUS(td->cancelstatus))
    {
        // set _kexit() as return thread function
        set_exit_addr(td);

        // move thread to active queue
        remqueue((struct Gqueue *)td);
        _Add_ready(td);

        // If the timer was armed, need to remove it
        if ((td->td_timeout.state & TIMER_ARMED) != 0U)
        {
            remqueue ((struct Gqueue *) &td->td_timeout);
            td->td_timeout.state &= ~TIMER_ARMED;
        }
    }

    i_restore(ps);
    return 0;
}


/**
 * @} end pthread group
 */
