/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"

void _kexit(void *ret);


/**
 * @addtogroup pthread_group pthread
 * @{
 */


 /**
 *  @brief Thread termination.
 *
 *  @param [in] status: pointer to the status of thread end.
 *
 *  @retval no return value.
 */
void pthread_exit (void *status)
{
    // Log this kernel function
    DLOG_PTHREAD_EXIT_INSERT;
    _kexit(status);
}


/**
 * @} end pthread group
 */


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */

/* SAM Should this be abnormal_exit? */
/**
 *  @brief Thread unblock.
 *
 *  @param [in] td: thread identificator.
 *
 *  @retval no return value.
 */
static void unblock (Thread * td)
{
    td->td_correspondent = 0;     /* return failed */
    td->td_errno = ETHREADDIED;
    remqueue((struct Gqueue *)td);
    _Add_ready (td);
}


/**
 *  @brief Thread exit.
 *
 *  @param [in] ret: pointer to the status of thread end.
 *
 *  @retval no return value.
 */
void _kexit (void *ret)
{
    int i;
    struct _pthread_cleanup_handler *cleanup;

    /* Call cleanup handlers */
    while (_Active->cleanup_stack != NULL)
    {
        cleanup = (struct _pthread_cleanup_handler *)_Active->cleanup_stack;
        _Active->cleanup_stack = (void*)cleanup->prev;
        cleanup->routine(cleanup->arg);
    }

    (void)i_disable();              /* _Dispatch will enable to next threads
                                    level */

    /* Remove ourselves from ready queue. */
    remqueue((struct Gqueue *) _Active);

    if (_Active->td_rd != NULL)
    {
        register RendezDescriptor *rd;

        rd = _Active->td_rd;

        /* Unblock anyone waiting blocked on us. */
        for (i = 0; i < rd->td_nports; ++i)
            while (!queueIsEmpty(&rd->td_sendq[i]))
                unblock((Thread *)rd->td_sendq[i].q_head);
        while (!queueIsEmpty(&rd->td_replyq))
            unblock((Thread *)rd->td_replyq.q_head);

        /* free the rendez descriptor */
        kfree(KA_RD, rd);
    }

    _Active->exit_return = ret;

    /*
     * Here, some of compilers can generate a warning like "the order of volatile accesses
     * is undefined in this statement".
     *
     * "_Active" is a volatile variable and can be changed at any time in interrupts or other tasks.
     * The code below makes an operation under two variables pointed by "_Active".
     * So is it possible a situation when they will be changed during this operation.
     * To avoid it the code is executed inside a critical section i_disable()/i_restore().
     */
    // if this thread ending is expected
    if (!queueIsEmpty((struct Gqueue *) &_Active->join_queue))
    {
        register Thread *td;
        td = (Thread *) _Active->join_queue.q_head;
        remqueue ((struct Gqueue *) td);
        _Add_ready (td);

        _Active->td_state = DEAD;
        kfree(KA_STACK, _Active->td_stackbase);
    }

    // if this thread non joinable
    else if (_Active->detachstate == PTHREAD_CREATE_JOINABLE)
    {
        _Active->td_state = DEAD;
        kfree(KA_STACK, _Active->td_stackbase);
    }
    else
    {
        /* free up the stack and thread descriptor */
        _Active->td_id = UINT_MAX;
        kfree(KA_STACK, _Active->td_stackbase);
        TDfree(_Active);
    }

    /* and context switch ourselves outa here! We call Dispatch since _Active
     is no longer valid. */
    _Dispatch();
}


/**
 * @} end private kernel functions group
 */
