/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "kernel.h"


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


 /**
 *  @brief     Return bit number of the first bit set in mask.
 *
 *  @param[in] mask: mask value. Mask cannot be zero.
 *
 *  @return    bit number of the first bit set in mask.
 */
int find_first_bit(BitMask mask)
{
    register int bit;
    register BitMask bmask = mask;
    for (bit = 0; (bmask & 1U) == 0U; ++bit)
        bmask >>= 1U;
    return bit;
}



/**
 *  @brief     Remove thread from whichever queue it happens to be in.
 *
 *  @param[in] td: pointer to queue object.
 *
 *  @return    none.
 *
 *  @details   2007 IKM Allows calls when item not on any queue.
 *             You MUST initqueue() the thing when you create it, though!
 *             DO NOT OPTIMIZE THE FOLLOWING QUEUE ROUTINES.
 *             All attempts at optimization have failed with one compiler or another.
 */
void remqueue(struct Gqueue *td)
{
    register struct Gqueue *qp, *qn;

    qp = td->q_tail;
    qn = td->q_head;

    qn->q_tail = qp;
    qp->q_head = qn;
    td->q_head = td;
    td->q_tail = td;
}


/**
 *  @brief     Insert thread at the tail of the queue.
 *
 *  @param[in] td: pointer to inserted object.
 *  @param[in] qh: pointer to queue object.
 *
 *  @return    none.
 */
void insqueue(struct Gqueue *td, struct Gqueue *qh)
{
    register struct Gqueue *qp;

    qp = qh->q_tail;

    td->q_head = qh;
    td->q_tail = qp;
    qh->q_tail = td;
    qp->q_head = td;
}


/**
 *  @brief     Insert thread at the head of the queue.
 *
 *  @param[in] td: pointer to inserted object.
 *  @param[in] qh: pointer to queue object.
 *
 *  @return    none.
 */
void insqueue_head(struct Gqueue *td, struct Gqueue *qh)
{
    insqueue(td, qh->q_head);
}

/**
 *  @brief     Insert the thread in priority order onto a waiting queue.
 *
 *  @param[in] td: pointer to inserted object.
 *  @param[in] qh: pointer to queue object.
 *
 *  @return    none.
 *
 *  @details   The thread is inserted immediately
 *             after all the others of the same priority. This ensures that the head of the queue is always
 *             the highest priority thread that has been waiting the longest, as per Posix-2001 rules for
 *             threads that block on mq_send, mq_receive, and sem_wait.
 */
void insqueue_prio(struct Gqueue *td, struct Gqueue *qh)
{
    struct Gqueue *qp = qh->q_head;

    while ((qp != qh) && (((Thread *) qp)->td_myqueue > ((Thread *) td)->td_myqueue))
        qp = qp->q_head;

    insqueue(td, qp);
}


/**
 *  @brief     Move the thread from whichever queue it happens to be in to the new queue.
 *
 *  @param[in] td: pointer to inserted object.
 *  @param[in] qh: pointer to queue object.
 *
 *  @return    none.
 *
 *  @details   Really an optimization for 'remqueue(td); insqueue(td, qh);'.
 *
 *
 */
void movequeue(struct Gqueue *td, struct Gqueue *qh)
{
    register struct Gqueue *qp, *qn;

    /* remqueue(td) */
    qn = td->q_head;
    qp = td->q_tail;

    qn->q_tail = qp;
    qp->q_head = qn;

    /* insqueue(td, qh) */
    qp = qh->q_tail;

    td->q_head = qh;
    td->q_tail = qp;
    qh->q_tail = td;
    qp->q_head = td;
}

/**
 * @} end private kernel functions group
 */
