/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "../kernel/kernel.h"
#include "../kernel/_pthread.h"
#include "port.h"


/**
 * @addtogroup port_group port
 * @{
 */

void _kexit (void *ret);

/*
 * We do not need function _kexit_ret(void):
 * after return(param) - param to be in r0,
 * _kexit(void *ret) wait parameter in r0.
 * After return(), we can call _kexit() directly.
 */


/**
 *  @brief     Calculate used space in thread stack
 *
 *  @param[in]  td: pointer to the thread structure object
 *  @param[out] info: pointer to stack information structure
 *
 *  @retval     ENOSPC: error. The stack has overflowed.
 *  @retval     0: success
 *
 *  @note    To port this function to different platforms take into consideration
 *           stack type: it can grow up or down.
 */
int platform_stackinfo (Thread *td, stackinfo_t *info)
{
    register uint *p;
    register size_t i;

    p = (uint_32 *) td->td_stackbase;
    if (*p++ != STACK_MAGIC)
        return ENOSPC;
    if (info != NULL)
    {
        for (i = 0; *p == 0x53535353U; ++i)
            p++;
        info->si_used = info->si_size - (i * sizeof(int));
    }
    return 0;
}


/**
 *  @brief     Fill thread stack by initial values
 *
 *  @param[in]  td: pointer to the thread structure object
 *  @param[in]  arg: argument for running thread
 *  @param[in]  root_func: pointer to running thread
 *
 *  @return    pointer to filled stack
 */
char *platform_init_thread (Thread *td, void *arg, THREAD (*root_func)(void *arg))
{
    register unsigned int *sp;


    #define INITIAL_PSR          0x01000000U        //Bit 24 always 1
    #define INT_STATUS           0x00000000U        //Enable interrupt
    #define INITIAL_EXEC_RETURN  0xFFFFFFFDU        //Use non-floating-point state


    /* ARM Cortex-M stack grows down, so stack is the up of the stack. */
    sp = (unsigned int *) (td->td_stackbase + (td->td_stacksize - sizeof(int)));

    /*
     * Prepare stack for 8 bytes alignment (need for 'long long' and 'double').
     * We need to align to 4 bytes here, to account control word "0xDEADBEEF"
     */
    if (((unsigned int)sp & 0x00000007U) == 0U)
    {
        *sp = 0xAAAAAAAAU;
        sp--;
    }

    *sp = 0xDEADBEEFU;
    sp--;

    *sp = 0x12345678U;    /* Word to which the stack pointer will be left pointing after context restore. */
    sp--;


#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    /*
     * If FPU registers is used and context switch is occurred
     * at this place automatically will be stored FPU frame:
     *     dummy word
     *     FPSC register
     *     S15 - S0 registers
     */
    sp -= 18;
#endif // __ARMCORTEXM4F__ || __ARMCORTEXM7__


    /*--- HW saved Cortex-M3/M4 stack frame ---*/
    *sp = INITIAL_PSR;
    sp--;

    *sp = (unsigned int)root_func;  /* PC = starting function */
    sp--;

    *sp = (unsigned int)_kexit;     /* r14 = lr */
    sp--;

    *sp = 0xCCCCCCCCU;              // R12
    sp--;

    *sp = 0x33333333U;              // R3
    sp--;

    *sp = 0x22222222U;              // R2
    sp--;

    *sp = 0x11111111U;              // R1
    sp--;

    *sp = (unsigned int)arg;        // R0
    sp--;
    /*--------------------------------------*/


    /*
     * If FPU registers is used and context switch is occurred
     * at this place we will store FPU high registers:
     *     S16 - S31 registers
     */


    *sp = INT_STATUS;               // Current status of int enable/disable
    sp--;

    *sp = INITIAL_EXEC_RETURN;      // Current exception return status - R14
    sp--;

    *sp = 0xBBBBBBBBU;              // R11
    sp--;

    *sp = 0xAAAAAAAAU;              // R10
    sp--;

    *sp = 0x99999999U;              // R9
    sp--;

    *sp = 0x88888888U;              // R8
    sp--;

    *sp = 0x77777777U;              // R7
    sp--;

    *sp = 0x66666666U;              // R6
    sp--;

    *sp = 0x55555555U;              // R5
    sp--;

    *sp = 0x44444444U;              // R4

    *(unsigned int *)(td->td_stackbase) = STACK_MAGIC;

    return (char*)sp;
}


/**
 *  @brief     Sets in thread stack _kexit() as return function
 *             and PTHREAD_CANCELED as return value.
 *
 *  @param[in]  td: pointer to the thread structure object
 *
 *  @return    no return value.
 *
 *  @details   The function gets thread's saved current stack pointer
 *             and replaces r0 and PC values in the appropriate offsets.
 *             As an additional action the function writes an initial value
 *             in PSR register for the case if thread's stack was corrupted.
 *             See the function platform_init_thread() to calculate these offsets.
 */
#define LR_OFFSET       8
#define R0_OFFSET       10
#define PC_OFFSET       16
#define PSR_OFFSET      17
#define FPU_HIGH_LEN    16
void set_exit_addr(Thread *td)
{
    unsigned int *sp;
#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    unsigned int fpu_regs_not_saved;
#endif

    sp = (unsigned int *)td->td_stackpointer;

#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    /*
     * Read LR and check FPU register saved in stack or not:
     *     0xfffffffd - non-floating-point state (FPU registers not saved)
     *     0xffffffed - floating-point state (FPU registers saved)
     */
    fpu_regs_not_saved = *(sp+LR_OFFSET) & 0x00000010U;

    if (fpu_regs_not_saved != 0U)
    {
        *(sp+R0_OFFSET) = (unsigned int)PTHREAD_CANCELED;   // r0 - return value
        *(sp+PC_OFFSET) = (unsigned int)_kexit;             // PC - return function
        *(sp+PSR_OFFSET) = INITIAL_PSR;                     // PSR register - initial value
    }
    else
    {
        *(sp+PC_OFFSET+FPU_HIGH_LEN) = (unsigned int)PTHREAD_CANCELED;  // r0 - return value
        *(sp+PC_OFFSET+FPU_HIGH_LEN) = (unsigned int)_kexit;            // PC - return function
        *(sp+PSR_OFFSET+FPU_HIGH_LEN) = INITIAL_PSR;                    // PSR register - initial value
    }
#else
    *(sp+R0_OFFSET) = (int)PTHREAD_CANCELED;    // r0 - return value
    *(sp+PC_OFFSET) = (int)_kexit;              // PC - return function
    *(sp+PSR_OFFSET) = (int)INITIAL_PSR;        // PSR register - initial value

#endif
}


/**
 *  @brief     Returns mininum thread stack size
 *
 *  @param     none.
 *
 *  @return    mininum thread stack size
 */
unsigned int platform_get_stack_min(void)
{
    return PLATFORM_STACK_MIN;
}


/**
 *  @brief     Returns default thread stack size
 *
 *  @param     none.
 *
 *  @return    default thread stack size
 */
unsigned int platform_get_stack_default(void)
{
    return PLATFORM_STACK_DEFAULT;
}


/**
 * @} end port functions group
 */
