/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "../kernel/dlog_kernel_ev.h"
#include "../kernel/kernel.h"


// Register access
#define HWREG(x) (*((volatile unsigned long *)(x)))

#define NVIC_CPAC               0xE000ED88U  // Coprocessor Access Control
#define NVIC_CPAC_CP11_FULL     0x00C00000U  // Full Access
#define NVIC_CPAC_CP10_FULL     0x00300000U  // Full Access

#define NVIC_FPCC               0xE000EF34U  // Floating-Point Context Control
#define NVIC_FPCC_ASPEN         0x80000000U  // Automatic State Preservation Enable
#define NVIC_FPCC_LSPEN         0x40000000U  // Lazy State Preservation Enable

#define NVIC_INT_CTRL           0xE000ED04U  // Interrupt Control and State
#define NVIC_INT_CTRL_PEND_SV   0x10000000U  // PendSV Set Pending


/*
 * Current active thread
 */
Thread * volatile _Active = NULL;


/**
 * @addtogroup port_group port
 * @{
 */


/**
 *  @brief     Makes threads relinquish.
 *
 *  @param[in] ps: last interrupt status.
 *
 *  @return    no return value.
 *
 *  @details _Relinquish() is the core of the context switch. It allows a thread to
 *           give up the CPU, if there is another thread runnable. If the calling
 *           thread is the currently runnable thread, nothing happens. _Relinquish() is
 *           called with interrupts disabled.
 */
void _Relinquish (Status ps)
{
    if (_locklevel != 0U)
    {
        i_restore(ps);
        return;
    }

    // Log thread's state change
    DLOG_PTHREAD_CHANGE_STATE_INSERT;

    /* Only do switch if we're not the next runnable thread. */
    if (_Active != (Thread *)prio_q->q_head)
    {
        HWREG(NVIC_INT_CTRL) = NVIC_INT_CTRL_PEND_SV;    //Call PendSV
    }

    i_restore(ps);
}


/**
 *  @brief     Assign active task and run it.
 *
 *  @param     none.
 *
 *  @return    no return value.
 *
 *  @details The _Dispatch function is called during system startup, and during thread
 *           exit, when there is no active thread. It expects to be called disabled.
 */
void _Dispatch(void)
{
    // find non empty queue
    while (queueIsEmpty (prio_q))
    {
        prio_q--;
    }

    /* Make the head of the queue active. */
    _Active = (Thread *)prio_q->q_head;


#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    /* Ensure the VFP is enabled - it should be anyway. */
    HWREG(NVIC_CPAC) |= NVIC_CPAC_CP11_FULL | NVIC_CPAC_CP10_FULL;

  #ifdef LAZY_SAVE
    /* Lazy save always. */
    HWREG(NVIC_FPCC) |= NVIC_FPCC_ASPEN | NVIC_FPCC_LSPEN;
  #else
    /* Lazy save never. */
    HWREG(NVIC_FPCC) |= NVIC_FPCC_ASPEN;
    HWREG(NVIC_FPCC) &= ~NVIC_FPCC_LSPEN;
  #endif
#endif

    StartFirstTask();
}


/**
 * @} end port functions group
 */
