/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "../kernel/kernel.h"



/*
 * Try to get interrupt stack boundaries from a linker file
 */
#define __linker_stack_top      __initial_sp
#define __linker_stack_bottom   __heap_base
//#define __linker_stack_bottom          // not used in Keil

extern int __linker_stack_top;
extern int __linker_stack_bottom;
//extern int __linker_stack_size;        // not used for Keil


/*
 * Local stack variables
 */
static unsigned int __port_stack_top;
static unsigned int __port_stack_bottom;
static unsigned int __port_stack_size;

/*
 * User visible variable. Specifies ISR stack filling for debugging.
 */
int _isrStackFill = 0;

/*
 * Current interrupt nesting level.
 */
volatile unsigned int _isrlevel = 0;


/**
 * @addtogroup port_group port
 * @{
 */


/**
 *  @brief     Get current MSP
 *
 *  @param     none.
 *
 *  @return    pointer to current MSP
 */
static __asm void* getMSP(void)
{
    mrs r0, msp
    bx lr
}


/**
 *  @brief     Fill interrupt stack by 'I'.
 *
 *  @param     none.
 *
 *  @return    no return value.
 *
 *  @details The function main() works in the same stack. We need
 *           to fill stack starting from current SP position.
 *
 *  @note    To port this function to different platforms take into consideration
 *           stack type: it can grow up or down.
 */
void _isr_init(void)
{
    unsigned int i;
    char *_isr_stack;
    unsigned int _isr_stack_limit;


    __port_stack_top = (unsigned int)&__linker_stack_top;
    __port_stack_bottom = (unsigned int)&__linker_stack_bottom;
    __port_stack_size = __port_stack_top - __port_stack_bottom;

    // get current limit
    _isr_stack_limit = (unsigned int)getMSP() - __port_stack_bottom;

    /*
     * Some compilers optimize cycle 'for' below - they replace it to memset().
     * We need leave some memory in current stack for it.
     */
    _isr_stack_limit -= (unsigned int)sizeof(int) * 16U;

    // fill stack with 'I'
    _isr_stack = (char *)__port_stack_bottom;
    if (_isrStackFill != 0)
    {
        for (i = 0; i < _isr_stack_limit; i++)
        {
            _isr_stack[i] = 'I';
        }
    }

}


/**
 *  @brief     Get current interrupt nested level.
 *
 *  @param     none.
 *
 *  @return    current interrupt nested level
 *
 *  @note      In CubeMX kernel modifications the function returns only 1 or 0 depended on
 *             CPU is in the interrupt or doesn't.
 */
__asm unsigned int getisrlevel(void)
{
    mrs r0, ipsr
    cbz r0, ret
    movs r0, #1
ret
    bx lr
}


/**
 *  @brief     Returns interrupt stack information
 *
 *  @param[out] info: pointer to stack information structure
 *
 *  @return    no return value.
 *
 *  @note    To port this function to different platforms take into consideration
 *           stack type: it can grow up or down.
 */
void checkIstack(stackinfo_t *info)
{
    unsigned int *pointer;
    unsigned int used;

    info->si_current = (void *)getMSP();
    info->si_base = (void *)__port_stack_bottom;
    info->si_size = (size_t)__port_stack_size;

    // stack grows down for ARM Cortex-M
    if (_isrStackFill != 0)
    {
        pointer = (unsigned int *)__port_stack_bottom;
        while (((unsigned int)pointer < __port_stack_top) && (*pointer == 0x49494949U))   // 'IIII'
            pointer++;

        used = __port_stack_top - (unsigned int)pointer;
        info->si_used = (size_t)used;
    }
    else
        info->si_used = 0;
}


/**
 *  @brief     Disable global interrupt.
 *
 *  @param     none.
 *
 *  @return    interrupt status before disabling.
 */
__asm Status i_disable(void)
{
    mrs r0, primask
    cpsid i
    bx lr
}


/**
 *  @brief     Restore global interrupt.
 *
 *  @param[in] last interrupt status.
 *
 *  @return    no return value.
 */
void i_restore(Status ps)
{
    if (ps != 0U)
    {
        __asm volatile ("nop");
        __asm volatile ("cpsid i");
    }
    else
    {
        __asm volatile ("cpsie i");
        __asm volatile ("nop");
    }
}


/**
 *  @brief     Get current thread stack pointer register
 *
 *  @param     none.
 *
 *  @return    current thread stack pointer register
 */
__asm void* getSP(void)
{
    mrs r0, psp
    bx lr
}


/**
 * @} end port functions group
 */
