/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#ifndef _pthread_h_
#define _pthread_h_

#ifdef __cplusplus
extern "C" {
#endif

#include <bits/local_lim.h>

typedef unsigned int pthread_t;
typedef void *THREAD;

extern size_t pthreadStackDefault;
extern int pthreadStackFill;
extern int _isrStackFill;


typedef struct
{
    void *attrp;
} pthread_attr_t;

struct sched_param
{
    int sched_priority;
};

#ifndef _queue_struct
#define _queue_struct
struct Gqueue
{
    struct Gqueue *q_head;
    struct Gqueue *q_tail;
};
#endif


// Structures and defines for support mutex functions
typedef struct
{
    struct Gqueue queue;
    unsigned locked;
    int type;
    pthread_t owner;
} pthread_mutex_t;

typedef pthread_attr_t pthread_mutexattr_t;

#define PTHREAD_MUTEX_INITIALIZER(mutex)    {{(struct Gqueue *)&mutex, (struct Gqueue *)&mutex}, 0, PTHREAD_MUTEX_DEFAULT, 0}
#define PTHREAD_MUTEX_NORMAL        0
#define PTHREAD_MUTEX_ERRORCHECK    1
#define PTHREAD_MUTEX_RECURSIVE     2
#define PTHREAD_MUTEX_DEFAULT       PTHREAD_MUTEX_NORMAL


// Structures and defines for support barrier functions
typedef struct {
    struct Gqueue queue;
    unsigned count;
    unsigned int id;
} pthread_barrier_t;

typedef struct {
    int pshared;
} pthread_barrierattr_t;

#define PTHREAD_BARRIER_SERIAL_THREAD   20


// Defines for posix support pthread_join()
#define PTHREAD_CREATE_JOINABLE 0
#define PTHREAD_CREATE_DETACHED 1


// Structures and defines for support condvar functions
typedef struct
{
    struct Gqueue queue;
    unsigned int id;
} pthread_cond_t;

typedef pthread_attr_t pthread_condattr_t;

#define PTHREAD_COND_INITIALIZER(cond)    {{(struct Gqueue *)&cond, (struct Gqueue *)&cond}, (unsigned int)&cond}


/* Cleanup buffer */
struct _pthread_cleanup_handler
{
    void (*routine) (void *);              /* Function to call */
    void *arg;                             /* Argument */
    struct _pthread_cleanup_handler *prev; /* Prev cleanup functions */
};

/* Cancellation */
#define PTHREAD_CANCEL_ENABLE     0
#define PTHREAD_CANCEL_DISABLE    1

#define PTHREAD_CANCEL_DEFERRED        0
#define PTHREAD_CANCEL_ASYNCHRONOUS    1

#define PTHREAD_CANCELED ((void *) -1)


/* pthread functions */
int pthread_create(pthread_t *new_thread, const pthread_attr_t *attr,
                    void *(*start_routine)(void *arg), void *arg);
void pthread_exit(void *status);
pthread_t pthread_self(void);
int pthread_equal(pthread_t t1, pthread_t t2);
int sched_yield(void);
int pthread_setschedparam(pthread_t target_thread, int policy,
                           const struct sched_param *param);
int pthread_getschedparam(pthread_t target_thread, int *policy,
                           struct sched_param *param);
int pthread_join(pthread_t thread, void **value_ptr);
int pthread_detach(pthread_t thread);

/* pthread attribute functions */
int pthread_attr_init(pthread_attr_t *attr);
int pthread_attr_destroy(pthread_attr_t *attr);
int pthread_attr_setstacksize(pthread_attr_t *attr, size_t stacksize);
int pthread_attr_getstacksize(const pthread_attr_t *attr,
                               size_t *stacksize);
int pthread_attr_setstackaddr(pthread_attr_t *attr, void *stackaddr);
int pthread_attr_getstackaddr(const pthread_attr_t *attr, void **stackaddr);
int pthread_attr_setschedparam(pthread_attr_t *attr,
                                const struct sched_param *param);
int pthread_attr_getschedparam(const pthread_attr_t * attr,
                                struct sched_param *param);
int pthread_attr_setdetachstate(pthread_attr_t *attr, int detachstate);
int pthread_attr_getdetachstate(const pthread_attr_t *attr, int *detachstate);
int pthread_getattr_np(pthread_t thread_id, pthread_attr_t *attr);

/* pthread mutex functions */
int pthread_mutex_init(pthread_mutex_t *mp,
                        const pthread_mutexattr_t *attr);
int pthread_mutex_destroy(pthread_mutex_t *mp);
int pthread_mutex_lock(pthread_mutex_t *mp);
int pthread_mutex_timedlock(pthread_mutex_t *mp, const struct timespec *abs_timeout);
int pthread_mutex_trylock(pthread_mutex_t *mp);
int pthread_mutex_unlock(pthread_mutex_t *mp);
int pthread_mutex_getlock_np(pthread_mutex_t *mp);
int pthread_mutexattr_init(pthread_mutexattr_t *attr);
int pthread_mutexattr_destroy(pthread_mutexattr_t *attr);
int pthread_mutexattr_settype(pthread_mutexattr_t *attr, int type);
int pthread_mutexattr_gettype(const pthread_mutexattr_t *attr, int *type);

/* pthread barrier functions */
int pthread_barrier_init(pthread_barrier_t *barrier,
                          const pthread_barrierattr_t *attr,
                          unsigned count);
int pthread_barrier_destroy(pthread_barrier_t *barrier);
int pthread_barrier_wait(pthread_barrier_t *barrier);

/* pthread cond functions */
int pthread_cond_init(pthread_cond_t *cond, const pthread_condattr_t *attr);
int pthread_cond_destroy(pthread_cond_t *cond);
int pthread_cond_wait(pthread_cond_t *cond, pthread_mutex_t *mutex);
int pthread_cond_timedwait(pthread_cond_t *cond, pthread_mutex_t *mutex,
                            const struct timespec *abstime);
int pthread_cond_signal(pthread_cond_t *cond);
int pthread_cond_broadcast(pthread_cond_t *cond);

int pthread_condattr_init(pthread_condattr_t *attr);
int pthread_condattr_destroy(pthread_condattr_t *attr);

int pthread_setprio(pthread_t thread, int prio);
int pthread_getprio(pthread_t thread, int *prio);

/* pthread cancel functions */
int pthread_setcancelstate(int state, int *oldstate);
int pthread_setcanceltype(int type, int *oldtype);
int pthread_cancel(pthread_t thread);
void pthread_testcancel(void);
int pthread_setcanceltype_np(pthread_t thread, int type);

/* pthread cleanup functions */
#define pthread_cleanup_push(routine, arg)               \
    {                                                    \
        struct _pthread_cleanup_handler _handler;        \
        __pthread_cleanup_push((routine), (arg), &_handler);

#define pthread_cleanup_pop(execute)                     \
        __pthread_cleanup_pop((execute), &_handler);     \
    }

void __pthread_cleanup_push(void (*routine)(void *arg), void *arg, struct _pthread_cleanup_handler *handler);
void __pthread_cleanup_pop(int execute, struct _pthread_cleanup_handler *handler);


/* extensions */

/* stack info returned by pthread_stackinfo */
typedef struct
{
    void *si_current;       /* current stack pointer */
    void *si_base;          /* stack base */
    size_t si_size;         /* stack size */
    size_t si_used;         /* high water mark. Only valid if FillStack was set. */
} stackinfo_t;

int pthread_stackinfo(pthread_t tid, stackinfo_t * info);

#ifdef __cplusplus
} //extern "C" {
#endif

#endif /* _pthread_h_ */
