/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_pthread.h"


struct _cond_cleanup_handler
{
    pthread_mutex_t *mutex;
    void *abstime;
};
void condvar_cleanup(void *arg);


/**
 * @addtogroup pthread_cond_group condition variables
 * @{
 */


/**
 *  @brief     Blocks on a condition variable and wait on a condition.
 *
 *  @param[in] cond: pointer to condition variable object.
 *  @param[in] mutex: pointer to the mutex object.
 *  @param[in] abstime: pointer to timespec structure with absolute time.
 *
 *  @retval    0: successful completion.
 *  @retval    EINVAL: the value specified by cond or mutex or abstime is invalid.
 *  @retval    ETIMEDOUT: the time specified by abstime to pthread_cond_timedwait() has passed.
 */
int pthread_cond_timedwait(pthread_cond_t *cond, pthread_mutex_t *mutex, const struct timespec *abstime)
{
    Status ps;
    struct _pthread_cleanup_handler handler;
    struct _cond_cleanup_handler condhandler;
    int oldtype;

    /* Check for cancellation */
    CANCELLATION_POINT();

    if ((!VALID_CONDVAR(cond)) || (mutex == NULL))
    {
        return EINVAL;
    }

    // Log this kernel function
    DLOG_PTHREAD_COND_WAIT_INSERT;

    ps = i_disable();

    if (abstime != NULL)
    {                           /* put on timer queue */
        struct itimerspec itimer;

        if ((abstime->tv_nsec < 0) || (abstime->tv_nsec >= 1000000000))
        {
            i_restore(ps);
            return EINVAL;
        }

        /*
         * The function can't just return ETIMEDOUT. We need to go through the function's routine:
         * unlock mutex, context switch and lock mutex.
         */
        if (abstime->tv_sec < CurrentTime.tv_sec)
        {
            _Active->td_state = TIMER_EXPIRED;
            goto cond_routine;
        }
        if ((abstime->tv_sec == CurrentTime.tv_sec) && (abstime->tv_nsec <= CurrentTime.tv_nsec))
        {
            _Active->td_state = TIMER_EXPIRED;
            goto cond_routine;
        }
        itimer.it_value.tv_sec = abstime->tv_sec;
        itimer.it_value.tv_nsec = abstime->tv_nsec;
        itimer.it_interval.tv_sec = 0;
        itimer.it_interval.tv_nsec = 0;
        __timer_settime((TMCB *)(&_Active->td_timeout), TIMER_ABSTIME, &itimer, NULL);
    }


    /* put myself in the wait queue */
    _Active->td_state = COND_BLOCKED;
    remqueue((struct Gqueue *)_Active);
    insqueue_prio((struct Gqueue *)_Active, (struct Gqueue *)&cond->queue);


cond_routine:
    // Unblock mutex
    if (!queueIsEmpty((struct Gqueue *)&mutex->queue))
    {
        register Thread *td;
        td = (Thread *)mutex->queue.q_head;
        remqueue((struct Gqueue *)td);
        _Add_ready(td);
    }
    else
        mutex->locked = 0;

    // prepare for cancellation cleanup
    condhandler.mutex = mutex;
    condhandler.abstime = (void*)abstime;
    /* __pthread_cleanup_push(condvar_cleanup, &condhandler, &handler); */
    handler.routine = condvar_cleanup;
    handler.arg = &condhandler;
    handler.prev = (struct _pthread_cleanup_handler *)_Active->cleanup_stack;
    _Active->cleanup_stack = (void*)&handler;


    //set async type
    __pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);

    _Relinquish(ps);

    //restore old type
    __pthread_setcanceltype(oldtype, NULL);

    // remove cancellation handler
    /* __pthread_cleanup_pop(0, &handler); */
    _Active->cleanup_stack = (void*)handler.prev;

    // lock mutex before exit
    if (_Active->td_state == TIMER_EXPIRED)
    {
        __pthread_mutex_lock(mutex);
        return ETIMEDOUT;
    }
    else
    {
        if (abstime != NULL)
            __timer_settime ((TMCB *)(&_Active->td_timeout), 0, NULL, NULL);    //disarm timer
        __pthread_mutex_lock(mutex);
        return 0;
    }
}

/**
 *  @brief     Blocks on a condition variable and wait on a condition.
 *
 *  @param[in] cond: pointer to condition variable object.
 *  @param[in] mutex: pointer to the mutex object
 *
 *  @retval    0: successful completion.
 *  @retval    EINVAL: the value specified by cond or mutex is invalid.
 */
int pthread_cond_wait(pthread_cond_t *cond, pthread_mutex_t *mutex)
{
    return pthread_cond_timedwait(cond, mutex, NULL);
}

/**
 * @} end condition variables group
 */

/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


 /**
 *  @brief     Condition variable cleanup handler.
 *
 *  @param[in] arg: pointer to cleanup structure.
 *
 *  @retval    none
 */
void condvar_cleanup(void *arg)
{
    struct _cond_cleanup_handler *condhandler =
        (struct _cond_cleanup_handler *)arg;

    // lock mutex before exit
    if (_Active->td_state == TIMER_EXPIRED)
    {
        __pthread_mutex_lock(condhandler->mutex);
    }
    else
    {
        if (condhandler->abstime != NULL)
            __timer_settime((TMCB *)(&_Active->td_timeout), 0, NULL, NULL);    //disarm timer
        __pthread_mutex_lock(condhandler->mutex);
    }
}

/**
 * @} end private kernel functions group
 */
