/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_time.h"


#if _POSIX_TIMERS > 0


/**
 * @addtogroup timer_group timer
 * @{
 */


/**
 *  @brief     Create the timer.
 *
 *  @param[in] clock_id: specified clock.
 *  @param[in] evp: points to a sigevent structure.
 *  @param[in] timerid: identificator of the timer.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int timer_create(clockid_t clock_id, struct sigevent *evp, timer_t *timerid)
{
    static struct sigevent DefaultEvp = { SIGEV_SIGPOST, 0, {0} };
    TMCB *tmcb;
    struct sigevent *event;

    if (clock_id != CLOCK_REALTIME)
        RETURN_ERRNO(EINVAL);

    if (evp == NULL)
        /* default it */
        event = &DefaultEvp;
    else
    {
        switch (evp->sigev_notify)
        {
            case SIGEV_NONE:
                break;
            case SIGEV_SIGPOST:
                if ((evp->sigev_signo < 0) || (evp->sigev_signo >= MR_SIG_MAX))
                    RETURN_ERRNO(EINVAL);
                break;
            default:
                RETURN_ERRNO(EINVAL);
                break;
        }
        event = evp;
    }

    tmcb = kalloc (KA_TIMER, sizeof (TMCB));
    if (tmcb == NULL)
        RETURN_ERRNO(EAGAIN);

    (void)memset(tmcb, 0, sizeof(TMCB));
    tmcb->timerid = (timer_t)tmcb;
    tmcb->td = _Active;

    // Log this kernel function
    DLOG_TIMER_CREATE_INSERT;

    (void)memcpy (&tmcb->event, event, sizeof(struct sigevent));
    if (timerid != NULL)
        *timerid = tmcb->timerid;

    /* Set up the Q so remqueue() doesn't barf */
    initqueue((struct Gqueue *)tmcb);

    return 0;
}


/**
 * @} end timer group
 */


#endif /* _POSIX_TIMERS */
