/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <semaphore.h>
#include <unison_app_config.h>


// Second task priority
#define TASK_PRIO 5

// Running timer
extern TIM_HandleTypeDef htim6;

// Used semaphore
sem_t sema;


THREAD Main(void *), Task(void *);

/*
 * First thread setup structure:
 *   - function name,
 *   - stack size,
 *   - priority.
 */
ConfigTableStruct ConfigTable = {
    Main, 1024, 4
};


/*
 * Main thread
 */
THREAD Main(void *args)
{
    pthread_t tid;
    struct timespec time;
    pthread_attr_t attr;
    struct sched_param myNewPriority;
    stackinfo_t info;


#if RTOS_DEBUG
    dir_register("Main thread", pthread_self(), TYPE_THREAD);
#endif
    xprintf("Start Main\n");

    // init semaphore
    sem_init(&sema, 0, 0);
#if RTOS_DEBUG
    dir_register("semaphore", (uint)&sema, TYPE_SEMAPHORE);
#endif

    // Setup system time
    time.tv_sec = 0; time.tv_nsec = 0;
    clock_settime(CLOCK_REALTIME, &time);

    /*
     * Run second thread
     */
    pthread_attr_init(&attr);
    myNewPriority.sched_priority = TASK_PRIO;
    pthread_attr_setschedparam(&attr, &myNewPriority);

    if(pthread_create(&tid, &attr, &Task, 0))
        xprintf("pthread_create failed\n");

    pthread_attr_destroy(&attr);


    // Run timer
    HAL_TIM_Base_Start_IT(&htim6);


    /*
     * Show current thread stack information
     */
    pthread_stackinfo (pthread_self(), &info);
    xprintf("Main() stack:\n");
    xprintf("  base: 0x%x\n", info.si_base);
    xprintf("  current: 0x%x\n", info.si_current);
    xprintf("  size: %d\n", info.si_size);
    xprintf("  used: %d\n\n", info.si_used);



    // Let timer interrupt do some work and show it stack information
    sleep(3);
    checkIstack (&info);
    xprintf("ISR stack:\n");
    xprintf("  base: 0x%x\n", info.si_base);
    xprintf("  current: 0x%x\n", info.si_current);
    xprintf("  size: %d\n", info.si_size);
    xprintf("  used: %d\n\n", info.si_used);



    /*
     * Infinite loop. Just send semaphore to Task()
     */
    while(1)
    {
        // Send semaphore
        xprintf("Main() send sema\n");
        if(sem_post(&sema) == -1)
            xprintf("sem_post failed\n");

        sleep(5);
    }
}


/*
 * Second thread. It waits semaphore from Main() thread and from interrupt
 */
THREAD Task(void *args)
{
    stackinfo_t info;
    struct timespec time;


#if RTOS_DEBUG
    dir_register("Task thread", pthread_self(), TYPE_THREAD);
#endif
    xprintf("Start Task\n");


    /*
     * Show current thread stack information
     */
    pthread_stackinfo (pthread_self(), &info);
    xprintf("Task() stack:\n");
    xprintf("  base: 0x%x\n", info.si_base);
    xprintf("  current: 0x%x\n", info.si_current);
    xprintf("  size: %d\n", info.si_size);
    xprintf("  used: %d\n\n", info.si_used);


    /*
     * Infinite loop. Just wait semaphore from Main() and interrupt
     */
    while(1)
    {
        if(sem_wait(&sema) == -1)
            xprintf("sem_wait failed\n");

        // Print system time
        clock_gettime(CLOCK_REALTIME, &time);
        xprintf("  Task got sema. Time %lu\n\n", (unsigned long)time.tv_sec);
    }

}



/**
  * @brief  Period elapsed callback in non blocking mode
  * @param  htim: pointer to a TIM_HandleTypeDef structure that contains
  *                the configuration information for TIM module.
  * @retval None
  */
void HAL_TIM_PeriodElapsedCallback(TIM_HandleTypeDef *htim)
{
    if (htim == &htim6)
    {
        // Send semaphore
        xprintf("TIM6 send sema\n");
        if(sem_post(&sema) == -1)
            xprintf("sem_post failed\n");
    }
}
