/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <main.h>
#include <unison_app_config.h>


static int RTOS_inited = 0;


void UnisonKernel_Start(void)
{
    /* Disable all interrupts */
    __disable_irq();

    /* 1 - fill task stack by 'S' during initialization for debug purpose */
    pthreadStackFill = 1;

    /* 1 - fill isr stack by 'I' during initialization for debug purpose */
    _isrStackFill = 1;

    /* initializes stack for interrupt, sizes = CSTACK, see .icf file for detail */
    _isr_init();

    /* heap for kernel malloc initialization */
    malloc_init();

    /* additional hardware initialization */
    hw_init();

    //LED_init();
    RTOS_inited = 1;

    /* Start RTOS */
    Nano_Start();
}




void hw_init(void)
{
    int iseterror;
    clockid_t  my_timer_clockid;
    struct timespec my_timespec;

    // processor specific initialization with clock
    // first setup the interrupt vector, clock parameters
    //and then setup the timer hardware
    my_timespec.tv_sec = 0;
    my_timespec.tv_nsec = CLOCK_TIME_PERIOD;
    my_timer_clockid = CLOCK_REALTIME;
    iseterror = clock_init(my_timer_clockid, &my_timespec);

    if(iseterror != 0)
    {
        xprintf("clock_init error\n");
    }


    /*
     * PendSV should have least priority
     */
    HAL_NVIC_SetPriority(PendSV_IRQn,0xF,0);

    /*
     * Set SysTick priority higher than PendSV
     */
    HAL_NVIC_SetPriority(SysTick_IRQn,0xE,0);
}



/*
 * RTOS System Tick handler
 */
static int timer_cnt = 0;
void HAL_SYSTICK_Callback(void)
{
    if (RTOS_inited)
        timer_tick(CLOCK_REALTIME);

    if(++timer_cnt > 1000/4)
    {
        timer_cnt = 0;
        HAL_GPIO_TogglePin(LED_GPIO_Port, LED_Pin);
    }
}



/*******************************************************************************
 * Unison debug output section
 ******************************************************************************/

#ifndef DEBUG_TO_BUF
#define DEBUG_TO_BUF    0 // 1 - xprintf will printout debug information to
                          //     debug memory buffer "debug_buffer"
                          // 0 - xprintf will printout debug information to
                          //     serial port (UART).
#endif

#if DEBUG_TO_BUF == 0
    extern UART_HandleTypeDef huart1;
#else
    #define DEBUG_BUFSIZE   512     // Size for "debug_buffer" in bytes
    uchar debug_buffer[DEBUG_BUFSIZE];
    static int debug_index = 0;
#endif



/*
 * kernel xprintf low level function to put one byte to debug console
 * or debug_buffer, depending on define DEBUG_TO_BUF
 */
void xputchar (const int c)
{
#if DEBUG_TO_BUF == 0
    uint8_t ch = (uint8_t)c;
    if (c == '\n')
    {
        ch = '\r';
        HAL_UART_Transmit(&huart1, (uint8_t *)&ch, 1, 1000);
    }
    ch = (uint8_t)c;
    HAL_UART_Transmit(&huart1, (uint8_t *)&ch, 1, 1000);
#else
    debug_buffer[debug_index]=(uchar)c;
    if (++debug_index >= DEBUG_BUFSIZE) debug_index = 0;
#endif
}
