/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#ifndef _sys_h_
#define _sys_h_

#include <stdint.h>
#include <limits.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/interrupt.h>
#include <unistd.h>
#include <time.h>
#include <pthread.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct
{
    THREAD (*rootStartRoutine) (void *);
    size_t rootStackSize;
    int rootPriority;
} ConfigTableStruct;

extern ConfigTableStruct ConfigTable;


// type for dir_lookup_type()
typedef struct dir_buf_entry_type
{
    struct dir_buf_entry_type *next_entry_ptr;//pointer to next entry
    uint value;            //value
    char *name;            //pointer to name
}dir_buf_type_t;

//type for threads utilization
typedef struct
{
    uint thread_id;
    char *thread_name;
    u_long util;
}thread_util_t;


/*----------------------------------------------------------------------*\
 *                                                                      *
 *                       SYSTEM CALL PROTOTYPES                         *
 *                                                                      *
\*----------------------------------------------------------------------*/

/*----- Board Support Routines -----*/
void hw_init(void);
void malloc_init(void);
void set_IdleThread_stack(size_t size);
void set_IdleThread_callback(void (*funk)(void));

void Nano_Start(void);

void *getSP(void);

/*----- Pools -------*/
#define MAXPOOLS        8
int pool_create(int poolno, unsigned long startaddr, int length);
int pool_destroy(int poolno);
void *pool_alloc(int poolno, size_t size);
int pool_free(int poolno, void *buf);


/*----- Partitions --*/
int pt_create(size_t blocksize, int numblocks);
int pt_destroy(int ptno);
void *pt_getblock(int ptno);
void pt_freeblock(int ptno, void *block);

/*----- Heap -----*/
int heap_create(unsigned long startaddr, int length);
void *heap_malloc(size_t size);
void heap_free(void *buf);
void *heap_realloc(void *buf, size_t size);
int heap_info(unsigned int *mem_free, unsigned int *chunks, unsigned int *max_chunk, unsigned int *min_chunk);


#if defined(_TMS320C30) || defined(_TMS320C40)
  /* No worries */
  #define ALIGN_LONG(n)     (n)
#elif defined(__YDFP__)
  #define ALIGN_LONG(n)     (((uint_32)(n) + 7UL) & ~(0x00000007UL))
#else
  /* For alignment on a long word boundry/size where (3UL) is not understood by the compiler */
  #define ALIGN_LONG(n)     (((size_t)(n) + 3UL) & ~(0x00000003UL))
#endif

/* polled io functions */
void xputchar(const int c);
void xputs(const char *s);
int xprintf(const char *format, ...);


/* Defines for Directory Services */
#define TYPE_UNKNOWN    0U
#define TYPE_THREAD     1U
#define TYPE_SERVER     2U
#define TYPE_SEMAPHORE  3U
#define TYPE_MUTEX      4U
#define TYPE_MQUEUE     5U
#define TYPE_BARRIER    6U
#define TYPE_TIMER      7U
#define TYPE_POOL       8U
#define TYPE_ENV        9U
#define TYPE_COND_VAR   10U

/* Directory Services */
int dir_register(const char *name, uint value, uint type);
int dir_deregister(const char *name);
int dir_lookup(const char *name, uint *valuep);
int dir_lookup_string(const char *name, uint *valuep, size_t len);
int dir_lookup_type(dir_buf_type_t *Dir_buf_ptr, uint type);

/* Threads ulization */
int threads_numb(void);
void threads_utilization_start(thread_util_t *threads_table, int threads_num);
void threads_utilization_stop(void);

/* lock/unlock task switching */
void Nano_lock(void);
void Nano_unlock(void);

/* Access to kernel internal variables*/
unsigned int getisrlevel(void);
unsigned int getlocklevel(void);

void checkIstack(stackinfo_t *info);


#ifdef __cplusplus
} //extern "C" {
#endif

#endif /* _sys_h_ */
