/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "_time.h"
#include "dlog_kernel_ev.h"
#include "kernel.h"


#if _POSIX_TIMERS > 0

/**
 * @addtogroup clock_group clock
 * @{
 */


/**
 *  @brief     Sets the specified clock.
 *
 *  @param[in] clockid: specified clock.
 *  @param[in] tp: pointer to timespec structure.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int clock_settime(clockid_t clockid, const struct timespec *tp)
{
    Status ps;
    register TMCB *tmcb;
    struct timespec diff;

    if ((clockid != CLOCK_REALTIME) || (tp->tv_nsec < 0) || (tp->tv_nsec >= NANOSECOND))
    {
        errno = EINVAL;
        return -1;
    }

    // Log this kernel function
    DLOG_CLOCK_SETTIME_INSERT;

    ps = i_disable();
    diff.tv_sec = tp->tv_sec - CurrentTime.tv_sec;
    diff.tv_nsec = tp->tv_nsec - CurrentTime.tv_nsec;

    CurrentTime.tv_sec = tp->tv_sec;
    CurrentTime.tv_nsec = tp->tv_nsec;

    // If running not absolute timers are exist
    tmcb = (TMCB *)TimerQ.q_head;

    while (tmcb != (TMCB *)&TimerQ)
    {
      if ((tmcb->state & TIMER_ABS) == 0U)
      {
          tmcb->timeout.tv_sec += diff.tv_sec;
          tmcb->timeout.tv_nsec += diff.tv_nsec;

          // Time forward
          if (tmcb->timeout.tv_nsec >= NANOSECOND)
          {
              tmcb->timeout.tv_nsec -= NANOSECOND;
              tmcb->timeout.tv_sec++;
          }

          // Time back
          if (tmcb->timeout.tv_nsec < 0)
          {
              tmcb->timeout.tv_nsec += NANOSECOND;
              tmcb->timeout.tv_sec--;
          }
      }
      tmcb = (TMCB *)(((struct Gqueue *)tmcb)->q_head);
    }


    i_restore(ps);

    return 0;
}


/**
 * @} end clock group
 */

#endif /* _POSIX_TIMERS */
