/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"



/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */

 /**
 *  @brief     Allocate memory for rendezvous descriptor.
 *
 *  @param[in] nports: number of ports.
 *
 *  @retval    0: no available memory.
 *  @retval    RendezDescriptor: pointer to rendezvous descriptor.
 */
static RendezDescriptor *RDalloc(int nports)
{
    register RendezDescriptor *rd;
    size_t size;
    int i;

    size = sizeof (RendezDescriptor) + (((size_t)nports - (size_t)1) * sizeof(struct Gqueue));
    rd = kalloc (KA_RD, size);
    if (rd == NULL)
        return NULL;
    (void)memset (rd, 0, size);

    rd->td_nports = nports;

    initqueue(&rd->td_replyq);
    for (i = 0; i < nports; ++i)
        initqueue(&rd->td_sendq[i]);

    return rd;
}

/**
 * @} end private kernel functions group
 */



/**
 * @addtogroup rendezvous_group rendezvous
 * @{
 */


/**
 *  @brief Initializes ports for a thread.
 *
 *  @param[in] nports: defines the number of ports to create.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int mr_init(int nports)
{
    if (_Active->td_rd != NULL)
        RETURN_ERRNO(EEXIST);
    if ((nports <= 0) || (nports > 31))
        RETURN_ERRNO(EINVAL);
    _Active->td_rd = RDalloc (nports);
    if (_Active->td_rd == NULL)
        RETURN_ERRNO(ENOMEM);

    // Log this kernel function
    DLOG_MR_INIT_INSERT;

    return 0;
}

/**
 * @} end rendezvous group
 */
