/* ----------------------------------------------------------------------
 * $Date:        5. February 2013
 * $Revision:    V1.02
 *
 * Project:      CMSIS-RTOS API
 * Title:        cmsis_os.c
 *
 * Version 0.02
 *    Initial Proposal Phase
 * Version 0.03
 *    osKernelStart added, optional feature: main started as thread
 *    osSemaphores have standard behavior
 *    osTimerCreate does not start the timer, added osTimerStart
 *    osThreadPass is renamed to osThreadYield
 * Version 1.01
 *    Support for C++ interface
 *     - const attribute removed from the osXxxxDef_t typedef's
 *     - const attribute added to the osXxxxDef macros
 *    Added: osTimerDelete, osMutexDelete, osSemaphoreDelete
 *    Added: osKernelInitialize
 * Version 1.02
 *    Control functions for short timeouts in microsecond resolution:
 *    Added: osKernelSysTick, osKernelSysTickFrequency, osKernelSysTickMicroSec
 *    Removed: osSignalGet
 *
 *
 *----------------------------------------------------------------------------
 *
 * Portions Copyright  2016 STMicroelectronics International N.V. All rights reserved.
 * Portions Copyright (c) 2013 ARM LIMITED
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - Neither the name of ARM  nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *---------------------------------------------------------------------------*/

#include <sys.h>
#include <string.h>
#include "pthread.h"
#include "stdint.h"
#include "errno.h"
#include "cmsis_os.h"

/*
 * ARM Compiler 4/5
 */
#if   defined ( __CC_ARM )

  #define __ASM            __asm
  #define __INLINE         __inline
  #define __STATIC_INLINE  static __inline

  #include "cmsis_armcc.h"

/*
 * GNU Compiler
 */
#elif defined ( __GNUC__ )

  #define __ASM            __asm                                      /*!< asm keyword for GNU Compiler          */
  #define __INLINE         inline                                     /*!< inline keyword for GNU Compiler       */
  #define __STATIC_INLINE  static inline

  #include "cmsis_gcc.h"

/*
 * IAR Compiler
 */
#elif defined ( __ICCARM__ )

  #ifndef   __ASM
    #define __ASM                     __asm
  #endif
  #ifndef   __INLINE
    #define __INLINE                  inline
  #endif
  #ifndef   __STATIC_INLINE
    #define __STATIC_INLINE           static inline
  #endif

  #include <cmsis_iar.h>
#endif

#define tskIDLE_PRIORITY 1
#define portBASE_TYPE    int

/* Convert from CMSIS type osPriority to Unison priority number */
static portBASE_TYPE makeUnisonPriority (osPriority priority)
{
    portBASE_TYPE fpriority = tskIDLE_PRIORITY;

    if (priority != osPriorityError) {
        int different = (int)priority - (int)osPriorityIdle;
        fpriority += (portBASE_TYPE)different;
    }

    return fpriority;
}

/* Convert from Unison priority number to CMSIS type osPriority */
static osPriority makeCmsisPriority (portBASE_TYPE fpriority)
{
    osPriority priority = osPriorityError;

    int os_diff = (int)fpriority - (int)tskIDLE_PRIORITY;
    int cmsis_range = (int)osPriorityRealtime - (int)osPriorityIdle;

    if (os_diff <= cmsis_range) {
        int tmp = (int)osPriorityIdle + os_diff;
        priority = (osPriority)tmp;
    }

    return priority;
}

/*********************** Kernel Control Functions *****************************/
/**
* @brief  Initialize the RTOS Kernel for creating objects.
* @retval status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osKernelInitialize shall be consistent in every CMSIS-RTOS.
*/
osStatus osKernelInitialize (void){
    return osOK;
}

/**
* @brief  Start the RTOS Kernel with executing the specified thread.
* @param  thread_def    thread definition referenced with \ref osThread.
* @param  argument      pointer that is passed to the thread function as start argument.
* @retval status code that indicates the execution status of the function
* @note   MUST REMAIN UNCHANGED: \b osKernelStart shall be consistent in every CMSIS-RTOS.
*/
osStatus osKernelStart (void)
{
    if (getisrlevel() != 0U)
        return osErrorISR;
    extern void UnisonKernel_Start(void);
    UnisonKernel_Start();
    return osOK;
}

/**
* @brief  Check if the RTOS kernel is already started
* @param  None
* @retval Not support
* @note  MUST REMAIN UNCHANGED: \b osKernelRunning shall be consistent in every CMSIS-RTOS.
*/
int32_t osKernelRunning(void)        //Not support
{
    return 0;
}

#if (defined (osFeature_SysTick)  &&  (osFeature_SysTick != 0))     // System Timer available
/**
* @brief  Get the value of the Kernel SysTick timer
* @param  None
* @retval None
* @note   MUST REMAIN UNCHANGED: \b osKernelSysTick shall be consistent in every CMSIS-RTOS.
*/
uint32_t osKernelSysTick(void)
{
    struct timespec curr_time;
    uint64_t nanosec;

    if (getisrlevel() != 0U)
    return 0;

    (void)clock_gettime(CLOCK_MONOTONIC, &curr_time);

    nanosec = (uint64_t)curr_time.tv_sec * (uint64_t)NANOSECOND;
    nanosec += (uint64_t)curr_time.tv_nsec;
    return (uint32_t)(nanosec / (uint64_t)_POSIX_CLOCKRES_MIN);
}
#endif    // System Timer available
/*********************** Thread Management *****************************/
/**
* @brief  Create a thread and add it to Active Threads and set it to state READY.
* @param  thread_def    thread definition referenced with \ref osThread.
* @param  argument      pointer that is passed to the thread function as start argument.
* @retval thread ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osThreadCreate shall be consistent in every CMSIS-RTOS.
*/
osThreadId osThreadCreate (const osThreadDef_t *thread_def, void *argument)
{
    pthread_t thread;
    pthread_attr_t attr;
    struct sched_param myNewPriority;

    if (getisrlevel() != 0U)
        return NULL;

    if (thread_def == NULL){
        return NULL;
    }

    if (thread_def->tpriority == osPriorityError){
        return NULL;
    }

    if (pthread_attr_init(&attr) != 0)
        return NULL;

    myNewPriority.sched_priority = makeUnisonPriority(thread_def->tpriority);
    (void)pthread_attr_setschedparam(&attr, &myNewPriority);
    (void)pthread_attr_setstacksize(&attr, thread_def->stacksize);
    (void)pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

    if (pthread_create(&thread, &attr, (void *(*)(void *arg))thread_def->pthread, argument) != 0)
    {
        return NULL;
    }

    (void)pthread_attr_destroy(&attr);

    (void)pthread_setcanceltype_np(thread, PTHREAD_CANCEL_ASYNCHRONOUS);

    return (osThreadId)thread;
}

/**
* @brief  Return the thread ID of the current running thread.
* @retval thread ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osThreadGetId shall be consistent in every CMSIS-RTOS.
*/
osThreadId osThreadGetId (void)
{
    pthread_t thread;

    if (getisrlevel() != 0U)
        return NULL;

    thread = pthread_self();

    return (osThreadId)thread;
}

/**
* @brief  Terminate execution of a thread and remove it from Active Threads.
* @param   thread_id   thread ID obtained by \ref osThreadCreate or \ref osThreadGetId.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osThreadTerminate shall be consistent in every CMSIS-RTOS.
*/
osStatus osThreadTerminate (osThreadId thread_id)
{
    if (getisrlevel() != 0U)
        return osErrorISR;

    if (thread_id == NULL){
        return osErrorParameter;
    }

    if (pthread_cancel((pthread_t)thread_id)!=0){
        return osErrorOS;
    }

    return osOK;
}

/**
* @brief  Pass control to next thread that is in state \b READY.
* @retval status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osThreadYield shall be consistent in every CMSIS-RTOS.
*/
osStatus osThreadYield (void)
{
    if (getisrlevel() != 0U)
        return osErrorISR;

    (void)sched_yield();

    return osOK;
}

/**
* @brief   Change priority of an active thread.
* @param   thread_id     thread ID obtained by \ref osThreadCreate or \ref osThreadGetId.
* @param   priority      new priority value for the thread function.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osThreadSetPriority shall be consistent in every CMSIS-RTOS.
*/
osStatus osThreadSetPriority (osThreadId thread_id, osPriority priority)
{
    int status;
    portBASE_TYPE priority_th;

    if (getisrlevel() != 0U)
        return osErrorISR;

    if (thread_id == NULL){
        return osErrorParameter;
    }

    if (priority == osPriorityError)
        return osErrorValue;

    priority_th = makeUnisonPriority(priority);
    status = pthread_setprio((pthread_t)thread_id, priority_th);
    if (status == EINVAL){
        return osErrorOS;
    }else if (status == ESRCH){
        return osErrorOS;
    }

    return osOK;
}

/**
* @brief   Get current priority of an active thread.
* @param   thread_id     thread ID obtained by \ref osThreadCreate or \ref osThreadGetId.
* @retval  current priority value of the thread function.
* @note   MUST REMAIN UNCHANGED: \b osThreadGetPriority shall be consistent in every CMSIS-RTOS.
*/
osPriority osThreadGetPriority (osThreadId thread_id)
{
    if (getisrlevel() != 0U)
        return osPriorityError;

    int prio;
    if ((pthread_getprio((pthread_t)thread_id, &prio))!=0){
        return     osPriorityError;
    }

    return makeCmsisPriority(prio);
}

/*********************** Generic Wait Functions *******************************/
/**
* @brief   Wait for Timeout (Time Delay)
* @param   millisec time sleep value
* @retval  status code that indicates the execution status of the function.
*/
osStatus osDelay (uint32_t millisec)
{
    uint32_t sec;
    uint32_t nsec;
    struct timespec delay_time;

    if (getisrlevel() != 0U)
        return osErrorISR;

    sec = millisec / 1000U;
    nsec = (millisec % 1000U) * 1000000U;
    delay_time.tv_sec = (time_t)sec;
    delay_time.tv_nsec = (long)nsec;
    if (delay_time.tv_nsec >= NANOSECOND)
    {
        delay_time.tv_sec++;
        delay_time.tv_nsec -= NANOSECOND;
    }

    if ((nanosleep (&delay_time, NULL)) != 0){
        return osErrorResource;
    }

    return osEventTimeout;
}

#if (defined (osFeature_Wait)  &&  (osFeature_Wait != 0)) /* Generic Wait available */
/**
* @brief  Wait for Signal, Message, Mail, or Timeout
* @param   millisec  timeout value or 0 in case of no time-out
* @retval  event that contains signal, message, or mail information or error code.
* @note   MUST REMAIN UNCHANGED: \b osWait shall be consistent in every CMSIS-RTOS.
*/
osEvent osWait (uint32_t millisec);

#endif  /* Generic Wait available */

/***********************  Timer Management Functions ***************************/
/**
* @brief  Create a timer.
* @param  timer_def     timer object referenced with \ref osTimer.
* @param  type          osTimerOnce for one-shot or osTimerPeriodic for periodic behavior.
* @param  argument      argument to the timer call back function.
* @retval  timer ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osTimerCreate shall be consistent in every CMSIS-RTOS.
*/
osTimerId osTimerCreate (const osTimerDef_t *timer_def, os_timer_type type, void *argument)
{
    (void)timer_def;
    (void)type;
    (void)argument;

    return NULL;
}

/**
* @brief  Start or restart a timer.
* @param  timer_id      timer ID obtained by \ref osTimerCreate.
* @param  millisec      time delay value of the timer.
* @retval  status code that indicates the execution status of the function
* @note   MUST REMAIN UNCHANGED: \b osTimerStart shall be consistent in every CMSIS-RTOS.
*/
osStatus osTimerStart (osTimerId timer_id, uint32_t millisec)
{
    (void)millisec;

    if (getisrlevel() != 0U)
        return osErrorISR;

    if (timer_id == NULL)
        return osErrorParameter;

    return osErrorOS;
}

/**
* @brief  Stop a timer.
* @param  timer_id      timer ID obtained by \ref osTimerCreate
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osTimerStop shall be consistent in every CMSIS-RTOS.
*/
osStatus osTimerStop (osTimerId timer_id)
{
    if (getisrlevel() != 0U)
        return osErrorISR;

    if (timer_id == NULL)
        return osErrorParameter;

    return osErrorOS;
}

/**
* @brief  Delete a timer.
* @param  timer_id      timer ID obtained by \ref osTimerCreate
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osTimerDelete shall be consistent in every CMSIS-RTOS.
*/
osStatus osTimerDelete (osTimerId timer_id)
{
    if (getisrlevel() != 0U)
        return osErrorISR;

    if (timer_id == NULL)
        return osErrorParameter;

    return osErrorOS;
}

/***************************  Signal Management ********************************/
/**
* @brief  Set the specified Signal Flags of an active thread.
* @param  thread_id     thread ID obtained by \ref osThreadCreate or \ref osThreadGetId.
* @param  signals       specifies the signal flags of the thread that should be set.
* @retval previous signal flags of the specified thread or 0x80000000 in case of incorrect parameters.
* @note   MUST REMAIN UNCHANGED: \b osSignalSet shall be consistent in every CMSIS-RTOS.
*/
int32_t osSignalSet (osThreadId thread_id, int32_t signals)
{
    (void)thread_id;
    (void)signals;

    return (int32_t)0x80000000U; /* Task Notification not supported */
}

/**
* @brief  Clear the specified Signal Flags of an active thread.
* @param  thread_id  thread ID obtained by \ref osThreadCreate or \ref osThreadGetId.
* @param  signals    specifies the signal flags of the thread that shall be cleared.
* @retval  previous signal flags of the specified thread or 0x80000000 in case of incorrect parameters.
* @note   MUST REMAIN UNCHANGED: \b osSignalClear shall be consistent in every CMSIS-RTOS.
*/
int32_t osSignalClear (osThreadId thread_id, int32_t signals)
{
    (void)thread_id;
    (void)signals;

    return (int32_t)0x80000000U; /* Task Notification not supported */
}

/**
* @brief  Wait for one or more Signal Flags to become signaled for the current \b RUNNING thread.
* @param  signals   wait until all specified signal flags set or 0 for any single signal flag.
* @param  millisec  timeout value or 0 in case of no time-out.
* @retval  event flag information or error code.
* @note   MUST REMAIN UNCHANGED: \b osSignalWait shall be consistent in every CMSIS-RTOS.
*/
osEvent osSignalWait (int32_t signals, uint32_t millisec)
{
    osEvent ret;

    (void)signals;
    (void)millisec;

    ret.status = osErrorOS;	/* Task Notification not supported */

    return ret;
}

/****************************  Mutex Management ********************************/
/**
* @brief  Create and Initialize a Mutex object
* @param  mutex_def     mutex definition referenced with \ref osMutex.
* @retval  mutex ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osMutexCreate shall be consistent in every CMSIS-RTOS.
*/
osMutexId osMutexCreate (const osMutexDef_t *mutex_def)
{
    pthread_mutexattr_t m_attr;
    pthread_mutex_t     *mutex;

    if (getisrlevel() != 0U)
        return NULL;

    if (mutex_def == NULL)
        return NULL;

    (void)pthread_mutexattr_init(&m_attr);
    (void)pthread_mutexattr_settype(&m_attr, PTHREAD_MUTEX_RECURSIVE);

    mutex = malloc(sizeof(pthread_mutex_t));
    if (mutex == NULL)
        return NULL;

    if (pthread_mutex_init(mutex, &m_attr) != 0){
        free(mutex);
        (void)pthread_mutexattr_destroy(&m_attr);
        return NULL;
    }
    (void)pthread_mutexattr_destroy(&m_attr);

    return (osMutexId)mutex;
}

/**
* @brief Wait until a Mutex becomes available
* @param mutex_id      mutex ID obtained by \ref osMutexCreate.
* @param millisec      timeout value or 0 in case of no time-out.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osMutexWait shall be consistent in every CMSIS-RTOS.
*/
osStatus osMutexWait (osMutexId mutex_id, uint32_t millisec)
{
    uint32_t sec;
    uint32_t nsec;
    struct timespec timeout;
    int status = 0;

    if (getisrlevel() != 0U)
        return osErrorISR;

    if (mutex_id == NULL) {
        return osErrorParameter;
    }

    if (millisec == osWaitForever){
        if (pthread_mutex_lock((pthread_mutex_t*)mutex_id)!=0){
            return osErrorOS;
        }
        return osOK;
    }

    if (millisec == 0U){
        status = pthread_mutex_trylock((pthread_mutex_t*)mutex_id);
        if (status != 0){
            if (status == EBUSY)
                return osErrorResource;
        }
        return osOK;
    }

    (void)clock_gettime(CLOCK_REALTIME, &timeout);

    sec = millisec / 1000U;
    nsec = (millisec % 1000U) * 1000000U;
    timeout.tv_sec += (time_t)sec;
    timeout.tv_nsec += (long)nsec;
    if (timeout.tv_nsec >= NANOSECOND){
        timeout.tv_sec++;
        timeout.tv_nsec -= NANOSECOND;
    }

    /* Lock Semaphore */
    if (pthread_mutex_timedlock((pthread_mutex_t*)mutex_id, &timeout) != 0){
        if (errno == ETIMEDOUT)
            return osErrorTimeoutResource;    // no mutex was available during timeout.
        else
            return osErrorOS;
    }

    return osOK;
}

/**
* @brief Release a Mutex that was obtained by \ref osMutexWait
* @param mutex_id      mutex ID obtained by \ref osMutexCreate.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osMutexRelease shall be consistent in every CMSIS-RTOS.
*/
osStatus osMutexRelease (osMutexId mutex_id)
{
    int getlock;

    if (getisrlevel() != 0U)
        return osErrorISR;

    if (mutex_id == NULL)
        return osErrorParameter;

    getlock = pthread_mutex_getlock_np((pthread_mutex_t*)mutex_id);

    if (getlock == 0)
        return osErrorResource;

    if (pthread_mutex_unlock((pthread_mutex_t*)mutex_id)!=0){
        return osErrorOS;
    }

    return osOK;
}

/**
* @brief Delete a Mutex
* @param mutex_id  mutex ID obtained by \ref osMutexCreate.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osMutexDelete shall be consistent in every CMSIS-RTOS.
*/
osStatus osMutexDelete (osMutexId mutex_id)
{
    if (getisrlevel() != 0U)
        return osErrorISR;

    if (mutex_id == NULL) {
        return osErrorParameter;
    }

    if (pthread_mutex_destroy((pthread_mutex_t*)mutex_id) != 0){
        return osErrorOS;
    }

    free((void *)mutex_id);
    return osOK;
}

/********************  Semaphore Management Functions **************************/

#if (defined (osFeature_Semaphore)  &&  (osFeature_Semaphore != 0))


/**
* @brief Create and Initialize a Semaphore object used for managing resources
* @param semaphore_def semaphore definition referenced with \ref osSemaphore.
* @param count         number of available resources.
* @retval  semaphore ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osSemaphoreCreate shall be consistent in every CMSIS-RTOS.
*/
osSemaphoreId osSemaphoreCreate (const osSemaphoreDef_t *semaphore_def, int32_t count)
{
    sem_t *sema;

    if (getisrlevel() != 0U)
        return NULL;

    if (semaphore_def == NULL)
        return NULL;

    sema = malloc(sizeof(sem_t));
    if (sema == NULL)
        return NULL;

    if (sem_init (sema, 0, (unsigned int)count) != 0){
        free(sema);
        return NULL;
    }

    return (osSemaphoreId)sema;
}

/**
* @brief Wait until a Semaphore token becomes available
* @param  semaphore_id  semaphore object referenced with \ref osSemaphore.
* @param  millisec      timeout value or 0 in case of no time-out.
* @retval  number of available tokens, or -1 in case of incorrect parameters.
* @note   MUST REMAIN UNCHANGED: \b osSemaphoreWait shall be consistent in every CMSIS-RTOS.
*/
int32_t osSemaphoreWait (osSemaphoreId semaphore_id, uint32_t millisec)
{
    int value;
    uint32_t sec;
    uint32_t nsec;
    struct timespec timeout;

    if (getisrlevel() != 0U)
        return (-1);

    if (semaphore_id == NULL){
        return (-1);
    }

    if (millisec == osWaitForever){
        if (sem_wait((sem_t*)semaphore_id) != 0){
            return -1;
        }
        (void)sem_getvalue((sem_t *)semaphore_id, &value);
        return (value+1);
    }

    if (millisec == 0U){
        if (sem_trywait((sem_t*)semaphore_id) != 0){
            if (errno == EINVAL)
                return -1;
            else
                return 0;    // no semaphore was available.
        }
        (void)sem_getvalue((sem_t *)semaphore_id, &value);
        return (value+1);
    }

    (void)clock_gettime(CLOCK_REALTIME, &timeout);

    sec = millisec / 1000U;
    nsec = (millisec % 1000U) * 1000000U;
    timeout.tv_sec += (time_t)sec;
    timeout.tv_nsec += (long)nsec;
    if (timeout.tv_nsec >= NANOSECOND)
    {
        timeout.tv_sec++;
        timeout.tv_nsec -= NANOSECOND;
    }

    /* Lock Semaphore */
    if (sem_timedwait((sem_t*)semaphore_id, &timeout) != 0){
        if (errno == ETIMEDOUT)
            return 0;    // no semaphore was available during timeout.
        else
            return -1;
    }
    (void)sem_getvalue((sem_t *)semaphore_id, &value);
    return (value+1);
}

/**
* @brief Release a Semaphore token
* @param  semaphore_id  semaphore object referenced with \ref osSemaphore.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osSemaphoreRelease shall be consistent in every CMSIS-RTOS.
*/
osStatus osSemaphoreRelease (osSemaphoreId semaphore_id)
{
    int sem;

    if (semaphore_id == NULL){
        return osErrorParameter;
    }

    sem = sem_post((sem_t*)semaphore_id);

    if (sem == EINVAL){
        return osErrorParameter;
    }

    if (sem != 0){
        return osErrorOS;
    }
    return osOK;
}

/**
* @brief Delete a Semaphore
* @param  semaphore_id  semaphore object referenced with \ref osSemaphore.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osSemaphoreDelete shall be consistent in every CMSIS-RTOS.
*/
osStatus osSemaphoreDelete (osSemaphoreId semaphore_id)
{
    if (getisrlevel() != 0U)
        return osErrorISR;

    if (semaphore_id == NULL)
        return  osErrorParameter;

    if (sem_destroy((sem_t*)semaphore_id)!=0){
        return osErrorOS;
    }

    free((void *)semaphore_id);
    return osOK;
}

#endif    /* Use Semaphores */

/*******************   Memory Pool Management Functions  ***********************/

#if (defined (osFeature_Pool)  &&  (osFeature_Pool != 0))

typedef struct os_pool_cb {
  int       pool;
  uint32_t pool_sz;
  uint32_t item_sz;
} os_pool_cb_t;

/**
* @brief Create and Initialize a memory pool
* @param  pool_def      memory pool definition referenced with \ref osPool.
* @retval  memory pool ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osPoolCreate shall be consistent in every CMSIS-RTOS.
*/
osPoolId osPoolCreate (const osPoolDef_t *pool_def)
{
    osPoolId pool_id;

    if ((pool_def == NULL) || (getisrlevel() != 0U))
        return NULL;

    pool_id = malloc(sizeof(*pool_id));
    int i = pt_create((size_t)pool_def->item_sz, (int)pool_def->pool_sz);

    pool_id->pool = i;
    pool_id->pool_sz = (uint32_t)pool_def->pool_sz;
    pool_id->item_sz = (uint32_t)pool_def->item_sz;

    return pool_id;
}

/**
* @brief Allocate a memory block from a memory pool
* @param pool_id       memory pool ID obtain referenced with \ref osPoolCreate.
* @retval  address of the allocated memory block or NULL in case of no memory available.
* @note   MUST REMAIN UNCHANGED: \b osPoolAlloc shall be consistent in every CMSIS-RTOS.
*/
void *osPoolAlloc (osPoolId pool_id)
{
    void *p;

    if (pool_id == NULL)
        return NULL;

    p = pt_getblock(pool_id->pool);
    return p;
}

/**
* @brief Allocate a memory block from a memory pool and set memory block to zero
* @param  pool_id       memory pool ID obtain referenced with \ref osPoolCreate.
* @retval  address of the allocated memory block or NULL in case of no memory available.
* @note   MUST REMAIN UNCHANGED: \b osPoolCAlloc shall be consistent in every CMSIS-RTOS.
*/
void *osPoolCAlloc (osPoolId pool_id)
{
    void *p;

    if (pool_id == NULL)
        return NULL;

    p = osPoolAlloc(pool_id);
    if (p != NULL)
        (void)memset(p, 0, (size_t)pool_id->item_sz);
    return p;
}

/**
* @brief Return an allocated memory block back to a specific memory pool
* @param  pool_id       memory pool ID obtain referenced with \ref osPoolCreate.
* @param  block         address of the allocated memory block that is returned to the memory pool.
* @retval  status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osPoolFree shall be consistent in every CMSIS-RTOS.
*/
osStatus osPoolFree (osPoolId pool_id, void *block)
{
    if ((pool_id == NULL) || (block == NULL))
        return osErrorParameter;

    pt_freeblock(pool_id->pool, block);
    if (block == NULL)
        return osErrorOS;

    free(pool_id);
    return osOK;
}

#endif   /* Use Memory Pool Management */

/*******************   Message Queue Management Functions  *********************/

#if (defined (osFeature_MessageQ)  &&  (osFeature_MessageQ != 0)) /* Use Message Queues */

/**
* @brief Create and Initialize a Message Queue
* @param queue_def     queue definition referenced with \ref osMessageQ.
* @param  thread_id     thread ID (obtained by \ref osThreadCreate or \ref osThreadGetId) or NULL.
* @retval  message queue ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osMessageCreate shall be consistent in every CMSIS-RTOS.
*/
osMessageQId osMessageCreate (const osMessageQDef_t *queue_def, osThreadId thread_id)
{
    struct mq_attr attr;
    static mqd_t mq;
    (void) thread_id;

    if ((queue_def == NULL) || (getisrlevel() != 0U))
        return NULL;

    attr.mq_flags    = 0;
    attr.mq_maxmsg    = (long)queue_def->queue_sz;
    attr.mq_msgsize    = (long)sizeof(uint32_t);//(long)queue_def->item_sz;

    mq = mq_open(queue_def->name, O_CREAT | O_RDWR, 0, &attr);

    return (osMessageQId)mq;
}

/**
* @brief Put a Message to a Queue.
* @param  queue_id  message queue ID obtained with \ref osMessageCreate.
* @param  info      message information.
* @param  millisec  timeout value or 0 in case of no time-out.
* @retval status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osMessagePut shall be consistent in every CMSIS-RTOS.
*/
osStatus osMessagePut (osMessageQId queue_id, uint32_t info, uint32_t millisec)
{
    uint32_t val;
    struct mq_attr mqstat;
    uint32_t sec;
    uint32_t nsec;
    struct timespec timeout;

    if ((getisrlevel() != 0U) && (millisec != 0U))
        return osErrorParameter;

    if (queue_id == NULL)
        return osErrorParameter;

    (void)mq_getattr((mqd_t)queue_id, &mqstat);
    val = info;

    if (millisec == osWaitForever){
        if (mq_send ((mqd_t)queue_id,(char*)&val, sizeof(val), 0) != 0){
            if (errno == EINVAL){
                return  osErrorParameter;
            }else if (errno == EMSGSIZE){
                return  osErrorResource;
            }else return osErrorOS;
        }
        return osOK;
    }

    if (millisec == 0U){
        if (mqstat.mq_curmsgs >= mqstat.mq_maxmsg){
            return osErrorResource;
        }else{
            if (mq_send ((mqd_t)queue_id,(char*)&val, sizeof(val), 0) != 0){
                if (errno == EINVAL){
                    return  osErrorParameter;
                }else if (errno == EMSGSIZE){
                    return  osErrorResource;
                }else return osErrorOS;
            }
            return osOK;
        }
    }

    sec = millisec / 1000U;
    nsec = (millisec % 1000U) * 1000000U;
    timeout.tv_sec = (time_t)sec;
    timeout.tv_nsec = (long)nsec;
    if (timeout.tv_nsec >= NANOSECOND)
    {
        timeout.tv_sec++;
        timeout.tv_nsec -= NANOSECOND;
    }

    errno = 0;
    if (mq_timedsend ((mqd_t)queue_id,(char*)&val, sizeof(val), 0, &timeout) == -1){
        if (errno == ETIMEDOUT){
            return  osErrorTimeoutResource;    // no msg was available during timeout.
        }else if (errno == EINVAL){
            return  osErrorParameter;
        }else if (errno == EMSGSIZE){
            return  osErrorResource;
        }else return osErrorOS;
    }

    return osOK;
}

/**
* @brief Get a Message or Wait for a Message from a Queue.
* @param  queue_id  message queue ID obtained with \ref osMessageCreate.
* @param  millisec  timeout value or 0 in case of no time-out.
* @retval event information that includes status code.
* @note   MUST REMAIN UNCHANGED: \b osMessageGet shall be consistent in every CMSIS-RTOS.
*/
osEvent osMessageGet (osMessageQId queue_id, uint32_t millisec)
{
    uint32_t sec;
    uint32_t nsec;
    struct timespec timeout;
    osEvent event;
    struct mq_attr mqstat;
    uint32_t val = 0;

    if ((getisrlevel() != 0U) && (millisec != 0U)){
        event.status = osErrorParameter;
        return event;
    }

    if (queue_id == NULL) {
        event.status = osErrorParameter;
        return event;
    }

    event.def.message_id = queue_id;
    event.value.p = NULL;

    (void)mq_getattr((mqd_t)queue_id, &mqstat);

    if (millisec == osWaitForever){
        if (mq_receive((mqd_t)queue_id, (char*)&val, sizeof(val), NULL) == -1){
            if ((errno == EBADF) || (errno == EMSGSIZE)){
                event.status = osErrorParameter;
                return event;
            }

        }
        event.status = osEventMessage;
        event.value.v = val;
        return event;
    }

    if (millisec == 0U)
    {
        if (mqstat.mq_curmsgs == 0)
        {
            event.status = osOK;
            return event;
        }
        else
        {
            if (mq_receive((mqd_t)queue_id, (char*)&val, sizeof(val), NULL) == -1)
            {
                if ((errno == EBADF) || (errno == EMSGSIZE))
                {
                    event.status = osErrorParameter;
                    return event;
                }
                else
                {
                    event.status = osErrorOS;
                    return event;
                }
            }
        }
        event.status = osEventMessage;
        event.value.v = val;
        return event;
    }

    sec = millisec / 1000U;
    nsec = (millisec % 1000U) * 1000000U;
    timeout.tv_sec = (time_t)sec;
    timeout.tv_nsec = (long)nsec;
    if (timeout.tv_nsec >= NANOSECOND)
    {
        timeout.tv_sec++;
        timeout.tv_nsec -= NANOSECOND;
    }

    if (mq_timedreceive ((mqd_t)queue_id, (char*)&val, sizeof(val), NULL, &timeout) == -1 ){
        if ((errno == EBADF) || (errno == EMSGSIZE) || (errno == EINVAL))
        {
            event.status = osErrorParameter;
            return event;
        }
        else if (errno == ETIMEDOUT)
        {
            event.status = osEventTimeout;
            return event;
        }
        else
        {
            event.status = osErrorOS;
            return event;
        }
    }

    event.value.v = val;
    event.status = osEventMessage;

    return event;
}

#endif     /* Use Message Queues */

//********************   Mail Queue Management Functions  ***********************/
#if (defined (osFeature_MailQ)  &&  (osFeature_MailQ != 0))  /* Use Mail Queues */


typedef struct os_mailQ_cb {
    const osMailQDef_t *queue_def;
    void *    handle;
    osPoolId pool;
} os_mailQ_cb_t;

/**
* @brief Create and Initialize mail queue
* @param  queue_def     reference to the mail queue definition obtain with \ref osMailQ
* @param   thread_id     thread ID (obtained by \ref osThreadCreate or \ref osThreadGetId) or NULL.
* @retval mail queue ID for reference by other functions or NULL in case of error.
* @note   MUST REMAIN UNCHANGED: \b osMailCreate shall be consistent in every CMSIS-RTOS.
*/
osMailQId osMailCreate (const osMailQDef_t *queue_def, osThreadId thread_id)
{

    (void) thread_id;

    struct mq_attr attr;
    static mqd_t mq;

    if ((queue_def == NULL) || (getisrlevel() != 0U))
        return NULL;

    attr.mq_flags    = 0;
    attr.mq_maxmsg    = (long)queue_def->queue_sz;
    attr.mq_msgsize    = (long)sizeof(uint32_t);//(long)queue_def->item_sz;

    mq = mq_open(queue_def->name, O_CREAT | O_RDWR, 0, &attr);

    osPoolDef_t pool_def = {queue_def->queue_sz, queue_def->item_sz, NULL};

    /* Create a mail queue control block */

    *(queue_def->cb) = malloc(sizeof(struct os_mailQ_cb));

    if (*(queue_def->cb) == NULL) {
        return NULL;
    }
    (*(queue_def->cb))->queue_def = queue_def;

    /* Create a queue*/
    (*(queue_def->cb))->handle = (void*)mq;


    if ((*(queue_def->cb))->handle == NULL) {
        free(*(queue_def->cb));
        return NULL;
    }

    /* Create a mail pool */
    (*(queue_def->cb))->pool = osPoolCreate(&pool_def);
    if ((*(queue_def->cb))->pool == NULL) {
        //Delete queue//
        free(*(queue_def->cb));
        return NULL;
    }
    return *(queue_def->cb);

}

/**
* @brief Allocate a memory block from a mail
* @param  queue_id      mail queue ID obtained with \ref osMailCreate.
* @param  millisec      timeout value or 0 in case of no time-out.
* @retval pointer to memory block that can be filled with mail or NULL in case error.
* @note   MUST REMAIN UNCHANGED: \b osMailAlloc shall be consistent in every CMSIS-RTOS.
*/
void *osMailAlloc (osMailQId queue_id, uint32_t millisec)
{
    (void) millisec;
    void *p;

    if ((millisec != 0U) && (getisrlevel() != 0U))
        return NULL;

    if (queue_id == NULL) {
        return NULL;
    }

    p = osPoolAlloc(queue_id->pool);

    return p;
}

/**
* @brief Allocate a memory block from a mail and set memory block to zero
* @param  queue_id      mail queue ID obtained with \ref osMailCreate.
* @param  millisec      timeout value or 0 in case of no time-out.
* @retval pointer to memory block that can be filled with mail or NULL in case error.
* @note   MUST REMAIN UNCHANGED: \b osMailCAlloc shall be consistent in every CMSIS-RTOS.
*/
void *osMailCAlloc (osMailQId queue_id, uint32_t millisec)
{
    void *p;

    if ((millisec != 0U) && (getisrlevel() != 0U))
        return NULL;

    p = osMailAlloc(queue_id, millisec);
    if (p != NULL)
        (void)memset(p, 0, (size_t)queue_id->queue_def->item_sz);
    return p;
}

/**
* @brief Put a mail to a queue
* @param  queue_id      mail queue ID obtained with \ref osMailCreate.
* @param  mail          memory block previously allocated with \ref osMailAlloc or \ref osMailCAlloc.
* @retval status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osMailPut shall be consistent in every CMSIS-RTOS.
*/
osStatus osMailPut (osMailQId queue_id, void *mail)
{
    uint32_t val;

    if (queue_id == NULL){
        return osErrorParameter;
    }else if (mail == NULL){
        return osErrorValue;
    }

    mqd_t a = (mqd_t)queue_id->handle;
    val = (uint32_t)mail;

    if (mq_timedsend (a,(char*)&val, sizeof(val), 0, NULL) == -1){
        return osErrorOS;
    }

    return osOK;
}

/**
* @brief Get a mail from a queue
* @param  queue_id   mail queue ID obtained with \ref osMailCreate.
* @param millisec    timeout value or 0 in case of no time-out
* @retval event that contains mail information or error code.
* @note   MUST REMAIN UNCHANGED: \b osMailGet shall be consistent in every CMSIS-RTOS.
*/
osEvent osMailGet (osMailQId queue_id, uint32_t millisec)
{
    osEvent event;
    struct mq_attr mqstat;
    uint32_t sec;
    uint32_t nsec;
    struct timespec timeout;
    uint32_t val = 0;

    if ((getisrlevel() != 0U) && (millisec != 0U)){
        event.status = osErrorParameter;
        return event;
    }

    if (queue_id == NULL) {
        event.status = osErrorParameter;
        return event;
    }

    event.def.message_id = queue_id;
    event.value.p = NULL;

    (void)mq_getattr((mqd_t)queue_id->handle, &mqstat);

    if (millisec == osWaitForever)
    {
        if (mq_receive((mqd_t)queue_id->handle, (char*)&val, sizeof(val), NULL) == -1)
        {
            if ((errno == EBADF) || (errno == EMSGSIZE))
            {
                event.status = osErrorParameter;
                return event;
            }

        }
        event.status = osEventMail;
        event.value.v = val;
        return event;
    }

    if (millisec == 0U)
    {
        if (mqstat.mq_curmsgs == 0)
        {
            event.status = osOK;
            return event;
        }
        else
        {
            if (mq_receive((mqd_t)queue_id->handle, (char*)&val, sizeof(val), NULL) == -1)
            {
                if ((errno == EBADF) || (errno == EMSGSIZE))
                {
                    event.status = osErrorParameter;
                    return event;
                }
                else
                {
                    event.status = osErrorOS;
                    return event;
                }
            }
        }
        event.status = osEventMail;
        event.value.v = val;
        return event;
    }

    sec = millisec / 1000U;
    nsec = (millisec % 1000U) * 1000000U;
    timeout.tv_sec = (time_t)sec;
    timeout.tv_nsec = (long)nsec;
    if (timeout.tv_nsec >= NANOSECOND)
    {
        timeout.tv_sec++;
        timeout.tv_nsec -= NANOSECOND;
    }

    if (mq_timedreceive ((mqd_t)queue_id->handle, (char*)&val, sizeof(val), NULL, &timeout) == -1)
    {
        if ((errno == EBADF) || (errno == EMSGSIZE) || (errno == EINVAL))
        {
            event.status = osErrorParameter;
            return event;
        }
        else if (errno == ETIMEDOUT)
        {
            event.status = osEventTimeout;
            return event;
        }
        else
        {
            event.status = osErrorOS;
            return event;
        }
    }
    event.value.v = val;
    event.status = osEventMail;

    return event;
}

/**
* @brief Free a memory block from a mail
* @param  queue_id mail queue ID obtained with \ref osMailCreate.
* @param  mail     pointer to the memory block that was obtained with \ref osMailGet.
* @retval status code that indicates the execution status of the function.
* @note   MUST REMAIN UNCHANGED: \b osMailFree shall be consistent in every CMSIS-RTOS.
*/
osStatus osMailFree (osMailQId queue_id, void *mail)
{
    if (queue_id == NULL)
        return osErrorParameter;
    else if (mail == NULL)
        return osErrorValue;

    return osPoolFree(queue_id->pool, mail);
}
#endif  /* Use Mail Queues */
