/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "kernel.h"
#include "directory.h"
#include "dirstubs.h"


/*
 * Pointer to queue with registered threads
 */
TabEntry *tHead = NULL;


/**
 * @addtogroup misc_group misc
 * @{
 */

/**
 *  @brief     Register a name with a value and type.
 *
 *  @param[in] name: pointer to the variable with name.
 *  @param[in] value: thread identifier.
 *  @param[in] type: pre-defined system types.
 *
 *  @retval    0: error. Variable errno is set appropriately. (See errno.h)
 *  @retval    1: success. Name associated with a value.
 */
int dir_register(const char *name, uint value, uint type)
{
    register TabEntry *tab;
    Status ps;

    ps = i_disable();

    /* search for the string */
    for (tab = tHead; tab != NULL; tab = tab->tab_next)
        if (strcmp(name, tab->tab_name) == 0)
        {    /* name already exists */
            errno = EEXIST;
            i_restore(ps);
            return 0;
        }

    /* add the entry to head of list */
    tab = kalloc(KA_DIR, sizeof(TabEntry) + strlen(name) + 1U);
    if (tab == NULL)
    {
        errno = ENOSPC;
        i_restore(ps);
        return 0;
    }
    (void)strcpy(tab->tab_name, name);
    tab->tab_namelen = (uint)(strlen(name) + 1U);
    tab->tab_value = value;
    tab->tab_type = type;
    tab->tab_next  = tHead;
    tHead = tab;
    i_restore(ps);
    return 1;
}


/**
 *  @brief     Remove registration by name.
 *
 *  @param[in] name: pointer to the variable with name.
 *
 *  @retval    0: negative. Entry not found.
 *  @retval    1: success. Entry deleted.
 */
int dir_deregister(const char *name)
{
    register TabEntry *tab, *p;
    Status ps;

    ps = i_disable();

    /* search for the string */
    p = NULL;
    for(tab = tHead; tab != NULL; tab = tab->tab_next)
    {
        if (strcmp(name, tab->tab_name) == 0)
        {    /* found it - p points to previous entry if any */
            if (p != NULL)
                p->tab_next = tab->tab_next;
            else
                tHead = tab->tab_next;
            kfree(KA_DIR, tab);
            i_restore(ps);
            return 1;
        }

        p = tab;
    }
    i_restore(ps);
    return 0;
}


/**
 *  @brief     Search an id for a name.
 *
 *  @param[in] name: pointer to variable with name.
 *  @param[out]valuep: pointer to thread identifier.
 *
 *  @retval    0: negative. Entry not found.
 *  @retval    1: success. Entry found.
 */
int dir_lookup(const char *name, uint *valuep)
{
    register TabEntry *tab;
    Status ps;

    ps = i_disable();

    for(tab = tHead; tab != NULL; tab = tab->tab_next)
        if (strcmp(name, tab->tab_name) == 0)
        {    /* found it */
            if (valuep != NULL) *valuep = tab->tab_value;
            i_restore(ps);
            return 1;
        }
    i_restore(ps);
    return 0;
}


/**
 *  @brief     Search an id for a partial name.
 *
 *  @param[in] name: pointer to the variable with name.
 *  @param[in] valuep: pointer to thread identifier.
 *  @param[in] len: number of characters are compared.
 *
 *  @retval    0: negative. Entry not found.
 *  @retval    1: success. Entry found.
 */
int dir_lookup_string(const char *name, uint *valuep, size_t len)
{
    register TabEntry *tab;
    Status ps;

    ps = i_disable();

    for(tab = tHead; tab != NULL; tab = tab->tab_next)
    {
        if (len < ((size_t)tab->tab_namelen - (size_t)1))
            continue;
        if (strncmp(name, tab->tab_name, len) == 0)
        {    /* found it */
            if (valuep != NULL) *valuep = tab->tab_value;
            i_restore(ps);
            return 1;
        }
    }
    i_restore(ps);
    return 0;
}

/**
 *  @brief     Lookup all names and values associated with a given type.
 *
 *  @param[in] Dir_buf_ptr: pointer to object entry structure.
 *  @param[in] type: pre-defined system types.
 *
 *  @retval   int: Counter of entries found.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int dir_lookup_type(dir_buf_type_t *Dir_buf_ptr, uint type)
{
    register TabEntry *tab;
    dir_buf_type_t    *dir_b_ptr;
    dir_buf_type_t    *tmp_ptr;
    int entries_cnt = 0;    //counter of entries in chain

    Status ps;

    ps = i_disable();

    if (Dir_buf_ptr == NULL)
    {
        errno = ENOSPC;
        i_restore(ps);
        return -1;
    }
    dir_b_ptr = Dir_buf_ptr;

    for(tab = tHead; tab != NULL; tab = tab->tab_next)
    {
        if (tab->tab_type == type)
        {
            // found it
            if (dir_b_ptr != NULL)
            {
                if (entries_cnt != 0)
                {
                    //allocate memory for new dir_buf_entry
                    tmp_ptr = malloc(sizeof(dir_buf_type_t));
                    if (tmp_ptr == NULL)
                    {
                        errno = ENOSPC;
                        i_restore(ps);
                        return -1;
                    }
                    //move ptr to new entry
                    dir_b_ptr->next_entry_ptr = tmp_ptr;
                    dir_b_ptr = tmp_ptr;
                    //fill values
                    dir_b_ptr->value = tab->tab_value;
                    dir_b_ptr->name = tab->tab_name;
                }
                else
                {
                    //fill values
                    dir_b_ptr->value = tab->tab_value;
                    dir_b_ptr->name = tab->tab_name;
                }
                entries_cnt ++;
            }
        }
    }
    i_restore(ps);
    return entries_cnt;
}


/**
 * @} end misc group
 */
