/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"

/* The following adds more debugging in rn_retseg */
#define HEAP_BOUNDS        0
#define DEBUGON            0
#define VERBOSE            0


/* header block - this must be x4 bytes */
typedef struct header {
    unsigned int magic;
#define MAGIC_MALLOC    (unsigned int)0x12121212
#define MAGIC_FREE      (unsigned int)0x23232323
    unsigned int size;
    struct header *next;
} header, *headerp;

/* Start & End Bytes */
#define START_BYTES     (sizeof(header))
#define END_BYTES       (sizeof(unsigned))
/* x is a total number of bytes available, like from get_mem */
#define USABLE_BYTES(x) ((x) - (int)(START_BYTES + END_BYTES))
/* s is the number of bytes the user requested */
#define TOTAL_BYTES(s)  ((s) + START_BYTES + END_BYTES)
#define END_OFFSET(s)   ((s) + START_BYTES)
#define END_MAGIC(h)    *(unsigned *)((char *)(h) + START_BYTES + (h)->size)
#define RET_ADDR(ptr)   ((void *)((char *)(ptr) + START_BYTES))

#define check_magic(hdr, m)    (((hdr)->magic == (m)) && (END_MAGIC(hdr) == (m)))

/* pool header */
typedef struct {
    int inuse;
    header freelist;
    headerp lfree;
#if HEAP_BOUNDS
    char *start;
    char *end;
#endif
} PoolHeader;

static PoolHeader Pool_m = {0};


 /**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


 /**
 *  @brief     Set the memory with header and debug info
 *
 *  @param[in] ptr: pointer to start address of memory.
 *  @param[in] size: memory size.
 *  @param[in] flag: specific flag for memory (MAGIC_MALLOC or MAGIC_FREE).
 *
 *  @return    no return value.
 */
static void set_mem(char *ptr, unsigned int size, unsigned int flag)
{
    headerp hptr;
    unsigned *eptr;

    /* given the pointer to the start of mem, assign the magic numbers
    and the size. */
    hptr = (headerp)ptr;
    hptr->magic = flag;
    hptr->size = size;
    eptr = (unsigned *)((char *)ptr + END_OFFSET(size));
    *eptr = flag;
}


/**
 * @} end private kernel functions group
 */


/**
 *  @brief     Create heap memory
 *
 *  @param[in] startaddr: start address of the memory area.
 *  @param[in] length: length of the memory area.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int heap_create(unsigned long startaddr, int length)
{
    Status ps;
    unsigned long align_startaddr;
    int len;

    ps = i_disable();

    if (Pool_m.inuse != 0)
    {
        i_restore(ps);
        errno = EINVAL;
        return -1;
    }

    /* align region on 32-bit boundry */
/******************* NEEDS ATTENTION WAS SCREWING US UP*************/
    align_startaddr = ALIGN_LONG(startaddr);
/*******************************************************************/
    /* get the actual size */
    len = length - (length % 4);    // length must be divisible by 4
    len = USABLE_BYTES(len);

    if (len <= 0)
    {
        i_restore(ps);
        errno = EINVAL;
        return -1;
    }

    /* set the memory, i.e. give the block structure  */
    set_mem((char *)align_startaddr, (unsigned int)len, MAGIC_FREE);
    /* initialize the region header */
    Pool_m.inuse            = 1;
    Pool_m.freelist.magic   = 0;
    Pool_m.freelist.size    = 0;
    Pool_m.freelist.next    = (headerp)align_startaddr;
    Pool_m.lfree            = &Pool_m.freelist;
    ((headerp)align_startaddr)->next = &Pool_m.freelist;
#if HEAP_BOUNDS
    Pool_m.start            = NULL;
    Pool_m.end              = NULL;
#endif
    i_restore(ps);
    return 0;
}


/**
 *  @brief     Allocation memory from heap
 *
 *  @param[in] size: size allocation memory.
 *
 *  @return    a pointer to a block of at least size bytes
 *             or NULL if no available memory.
 */
void *heap_malloc(size_t size)
{
    headerp hdr_free, hdr_prev;
    Status ps;
    size_t align_size;

    if (size == 0U)
        return NULL;

    // Log this kernel function
    DLOG_MALLOC_INSERT;

    /* must be a x4 */
    /******** NEEDS ATTENTION ***********/
    align_size = ALIGN_LONG(size);
    /* search for a memory block big enough */
    ps = i_disable();
    hdr_prev = Pool_m.lfree;
    for (hdr_free = hdr_prev->next; ; hdr_free = hdr_free->next)
    {
#if DEBUGON
        if ((hdr_free != &(Pool_m.freelist)) && (check_magic(hdr_free, MAGIC_FREE) == 0))
        {
            (void)xprintf("malloc: BAD MAGIC NUMBER! CORRUPTED HEAP!\n");
            return NULL;
        }
#endif

        /* free->size is a count of the USABLE BYTES.
         * looking for first memory equal to or bigger.
         */
        if ((hdr_free->size == align_size) || (hdr_free->size >= TOTAL_BYTES(align_size)))
        {
            if (hdr_free->size == align_size)
            {
                /* exact match - just delink from free list */
                hdr_prev->next = hdr_free->next;
                /* set memory as alloced */
                set_mem((char *)hdr_free, hdr_free->size, MAGIC_MALLOC);
            }
            else
            {
                /* reduce size of current block so we can strip off the end. */
                hdr_free->size -= (unsigned int)TOTAL_BYTES(align_size);
                /* set the memory of the reduced free block */
                set_mem((char *)hdr_free, hdr_free->size, MAGIC_FREE);
                hdr_free = (headerp)((char *)hdr_free + TOTAL_BYTES(hdr_free->size));
                /* set the memory of the tail portion. */
                set_mem((char *)hdr_free, (unsigned int)align_size, MAGIC_MALLOC);
            }

            /* set previous ptr to the place before the just-malloced
             * memory so that if that memory gets freed right away,
             * we will be ready at that spot.
             */
            Pool_m.lfree = hdr_prev;

#if HEAP_BOUNDS
            if ((unsigned)hdr_free < (unsigned)Pool_m.start)
                Pool_m.start = (char *)hdr_free;

            if ((unsigned)hdr_free > (unsigned)Pool_m.end)
                Pool_m.end = (char *)hdr_free;
#endif

            i_restore(ps);

            return RET_ADDR(hdr_free);
        }

        if (hdr_free == Pool_m.lfree)
            /* Traversed entire free list. */
            break;

        hdr_prev = hdr_free;
    }

    errno = ENOMEM;
    i_restore(ps);
    return NULL;
}


/**
 *  @brief     Free allocation memory
 *
 *  @param[in] buf: pointer to allocation memory.
 *
 *  @return    no return value.
 */
void heap_free(void *buf)
{
    headerp hdr, hdr_free;
    Status ps;

    if (buf == NULL)
        return;

    hdr = (headerp)((char *)buf - START_BYTES);

    if (((uint_32)hdr & (uint_32)(sizeof(long) - 1U)) != (uint_32)0)
    {
#if VERBOSE
        (void)xprintf("free: misaligned buf %x\n", buf);
#endif
        return;
    }

#if HEAP_BOUNDS
    if (((char *)hdr < Pool_m.start) || ((char *)hdr > Pool_m.end))
    {
#if VERBOSE
        (void)xprintf("free: address %x not in Pool_m\n", buf);
#endif
        return;
    }
#endif

    /* check to make sure the pointer passed in has a MAGIC_MALLOC */
    if (hdr->magic != MAGIC_MALLOC)
    {
#if VERBOSE
        if (hdr->magic == MAGIC_FREE)
            (void)xprintf("free: address %x already freed\n", buf);
        else
            (void)xprintf("free: address %x bad magic\n", buf);
#endif
        return;
    }

    if (END_MAGIC(hdr) != MAGIC_MALLOC)
    {
#if VERBOSE
        (void)xprintf("free: address %x end magic overwritten\n", buf);
#endif
        return;
    }

    // Log this kernel function
    DLOG_FREE_INSERT;

    /* Find where this block fits in the list.
     * Should be in between two elements in the list.
     */
    ps = i_disable();
    for (hdr_free = Pool_m.lfree; !((hdr > hdr_free) && (hdr < hdr_free->next)); hdr_free = hdr_free->next)
        /* if free is greater then its next pointer, we are at the end
         * of the list. AND block header must be either greater then the end
         * or before the beginning.
         */
        if ((hdr_free >= hdr_free->next) && ((hdr > hdr_free) || (hdr < hdr_free->next)))
            break;

    /* if end of this block equals the start of the next, connect to upper */
    if (((char *)hdr + TOTAL_BYTES(hdr->size)) == (char *)hdr_free->next)
    {
        hdr->size += (unsigned int)TOTAL_BYTES(hdr_free->next->size);
        hdr->next = hdr_free->next->next;
        set_mem((char *)hdr, hdr->size, MAGIC_FREE);
    }
    else
    {
        set_mem((char *)hdr, hdr->size, MAGIC_FREE);
        hdr->next = hdr_free->next;
    }

    /* if end of previous block equals this block, connect to lower */
    if (((char *)hdr_free + TOTAL_BYTES(hdr_free->size)) == (char *) hdr)
    {
        hdr_free->size += (unsigned int)TOTAL_BYTES(hdr->size);
        hdr_free->next = hdr->next;
        set_mem((char *)hdr_free, hdr_free->size, MAGIC_FREE);
    }
    else
        hdr_free->next = hdr;

    Pool_m.lfree = hdr_free;
    i_restore(ps);
    return;
}


/**
 *  @brief     Try to shrink or expand current block of memory
 *
 *  @param[in] buf: pointer to allocation memory.
 *  @param[in] size: requested size for current block.
 *
 *  @return    a pointer to a current block with new size
 *             or NULL in case of fail.
 */
static void *heap_modify_block(void *buf, size_t size)
{
   headerp hdr, hdr_free, hdr_new, hdr_rest;
   unsigned int new_size;
   Status ps;

   hdr = (headerp)((char *)buf - START_BYTES);
   new_size = ALIGN_LONG(size);

   /* Find where this block fits in the list.
    * Should be in between two elements in the list.
    */
   ps = i_disable();
   for (hdr_free = Pool_m.lfree; !((hdr > hdr_free) && (hdr < hdr_free->next)); hdr_free = hdr_free->next)
       /* if free is greater then its next pointer, we are at the end
        * of the list. AND block header must be either greater then the end
        * or before the beginning.
        */
       if ((hdr_free >= hdr_free->next) && ((hdr > hdr_free) || (hdr < hdr_free->next)))
           break;

   /* try to shrink current block of memory */
   if (hdr->size >= TOTAL_BYTES(new_size))
   {
       unsigned int curr_size = hdr->size;

       hdr_new = hdr;
       hdr = (headerp)((char *)hdr + TOTAL_BYTES(new_size));    // shift current block header

       set_mem((char *)hdr_new, new_size, MAGIC_MALLOC);                        // mark new block as allocated
       set_mem((char *)hdr, curr_size - TOTAL_BYTES(new_size), MAGIC_MALLOC);    // mark shifter block as allocated


       /* if end of this block equals the start of the next, connect to upper */
       if (((char *)hdr + TOTAL_BYTES(hdr->size)) == (char *)hdr_free->next)
       {
           hdr->size += (unsigned int)TOTAL_BYTES(hdr_free->next->size);
           hdr->next = hdr_free->next->next;
           set_mem((char *)hdr, hdr->size, MAGIC_FREE);
       }
       else
       {
           set_mem((char *)hdr, hdr->size, MAGIC_FREE);
           hdr->next = hdr_free->next;
       }

       // insert freed block in free list
       hdr_free->next = hdr;

       // update global structure
       Pool_m.lfree = hdr_free;
       i_restore(ps);
       return RET_ADDR(hdr_new);
   }


   /*
    *  try to expand current block of memory
    */
   // check if next free block is physically near the current one
   if (((char *)hdr + TOTAL_BYTES(hdr->size)) == (char *)hdr_free->next)
   {
       headerp hdr_next = hdr_free->next;

       // check if space in two blocks is enough for new block of memory
       if ((hdr->size + TOTAL_BYTES(hdr_next->size)) >= TOTAL_BYTES(new_size))
       {
           unsigned int diff_size = new_size - hdr->size;
           unsigned int next_size = hdr_next->size;

           hdr_rest = (headerp)((char *)hdr_next + diff_size);            // shift next block memory header (which is connected with current block)
           hdr_rest->next = hdr_next->next;                                // set 'next' pointer as it was in non-shifter block
           set_mem((char *)hdr_rest, next_size - diff_size, MAGIC_FREE);    // mark block as free
           hdr_free->next = hdr_rest;                                    // link 'next' pointer to rest block

           set_mem((char *)hdr, hdr->size + diff_size, MAGIC_MALLOC);    // update current header by new size

           // update global structure
           Pool_m.lfree = hdr_free;
           i_restore(ps);
           return RET_ADDR(hdr);
       }
   }

   i_restore(ps);
   return NULL;
}

/**
 *  @brief     Reallocation size of current block of memory
 *
 *  @param[in] buf: pointer to allocation memory.
 *  @param[in] size: requested size for current block.
 *
 *  @return    a pointer to a current block with new size
 *             or NULL in case of fail.
 */
void *heap_realloc(void *buf, size_t size)
{
    headerp hdr;
    void *ptr;
    size_t copy_len;

    if (size == 0U)
    {
        heap_free(buf);
        return buf;
    }

    if (buf == NULL)
        return heap_malloc(size);

    /*
     * Check buffer pointer
     */
    hdr = (headerp)((char *)buf - START_BYTES);
    if (((uint_32)hdr & (uint_32)(sizeof(long) - 1U)) != (uint_32)0)
    {
#if VERBOSE
        (void)xprintf("realloc: misaligned buf %x\n", buf);
#endif
        return NULL;
    }

#if HEAP_BOUNDS
    if (((char *)hdr < Pool_m.start) || ((char *)hdr > Pool_m.end))
    {
#if VERBOSE
        (void)xprintf("realloc: address %x not in Pool_m\n", buf);
#endif
        return NULL;
    }
#endif

    /* check to make sure the pointer passed in has a MAGIC_MALLOC */
    if (hdr->magic != MAGIC_MALLOC)
    {
#if VERBOSE
        if (hdr->magic == MAGIC_FREE)
            (void)xprintf("realloc: address %x already freed\n", buf);
        else
            (void)xprintf("realloc: address %x bad magic\n", buf);
#endif
        return NULL;
    }

    if (END_MAGIC(hdr) != MAGIC_MALLOC)
    {
#if VERBOSE
        (void)xprintf("realloc: address %x end magic overwritten\n", buf);
#endif
        return NULL;
    }


    /*
     * If the new size less than current one and a difference between them is less than a buffer header,
     * we can't resize current buffer and make a new free buffer.
     * In this case we return the same buffer.
     */
    if ((hdr->size >= size) && ((hdr->size - size) < (START_BYTES + END_BYTES)))
        return buf;

    /* Try to shrink or expand current block */
    ptr = heap_modify_block(buf, size);
    if (ptr != NULL)
        return ptr;

    /* Allocate a new block */
    ptr = heap_malloc(size);
    if (ptr == NULL)
        return NULL;

    /* Copy current block content to the new block */
    if (size > hdr->size)
        copy_len = hdr->size;
    else
        copy_len = size;
    (void)memcpy(ptr, buf, copy_len);
    heap_free(buf);
    return ptr;
}

/**
 *  @brief     Get heap information
 *
 *  @param[out] mem_free: total free memory.
 *  @param[out] chunks: number of free memory buffers.
 *  @param[out] max_chunk: size of maximum free buffer.
 *  @param[out] min_chunk: size of minimum free buffer.
 *
 *  @return    always return 0.
 */
int heap_info(unsigned int *mem_free, unsigned int *chunks, unsigned int *max_chunk, unsigned int *min_chunk)
{
    headerp hdr_free;
    Status ps;
    unsigned int curr_free = 0;
    unsigned int curr_max = 0;
    unsigned int curr_min = SIZE_MAX;
    unsigned int curr_chunks = 0;

    ps = i_disable();
    for (hdr_free = &Pool_m.freelist; ; hdr_free = hdr_free->next)
    {
        /*
         * Pointer to first chunk
         */
        if (hdr_free->size != 0U)
        {
            curr_chunks++;
            curr_free += hdr_free->size;
#if VERBOSE
            (void)xprintf("free chunk: %d\n", hdr_free->size);
#endif
            if (hdr_free->size > curr_max)
                curr_max = hdr_free->size;
            if (hdr_free->size < curr_min)
                curr_min = hdr_free->size;
        }
        if (hdr_free->next == &Pool_m.freelist)
            /* Traversed entire free list. */
            break;
    }

    if (mem_free != NULL)
        *mem_free = curr_free;
    if (chunks != NULL)
        *chunks = curr_chunks;
    if (max_chunk != NULL)
        *max_chunk = curr_max;
    if (min_chunk != NULL)
        *min_chunk = curr_min;

    i_restore(ps);
    return 0;
}
