/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_time.h"



/**
 * @addtogroup rendezvous_group rendezvous
 * @{
 */

/**
 *  @brief     Receive message or signal from threads
 *
 *  @param[out] rqst: pointer to area receiving a message.
 *  @param[in]  portmask: 32 bit port mask indicating which ports the thread is willing to accept a message through.
 *  @param[in]  sigmask: implements part of the selective accept mechanism.
 *  @param[out] outport: actual port or signal received.
 *  @param[in]  timeout: time that the caller is willing to wait before the system should indicate that the no signal or message was received.
 *
 *  @retval    pthread_t:  threads's id from which the message was received.
 *  @retval    INTERRUPT_ID: signal received.
 *  @retval    TIMEOUT_ID: timeout expired.
 *  @retval    0: call failed. Variable errno is set appropriately. (See errno.h)
 */
pthread_t mr_receive(void *rqst, BitMask portmask, BitMask sigmask,
            int *outport, const struct timespec *timeout)
{
    struct itimerspec sleeper;
    register RendezDescriptor *rd;
    BitMask mask;
    Status ps;

    rd = _Active->td_rd;

    //NOTE: && portmask>0 needs to be checked if mr_receive(0,0,0,0,FOREVER)
    //as well as in other parts of the code
    if (rd == NULL)
    {
        errno = ENOENT;
        return 0;
    }

    // Log this kernel function
    DLOG_MR_RECEIVE_INSERT;

    rd->td_portmask = portmask;
    _Active->td_sigmask = sigmask;


again:
    ps = i_disable();

    /*
     * Here, some of compilers can generate a warning like "the order of volatile accesses
     * is undefined in this statement".
     *
     * "_Active" is a volatile variable and can be changed at any time in interrupts or other tasks.
     * The code below makes an operation under two variables pointed by "_Active".
     * So is it possible a situation when they will be changed during this operation.
     * To avoid it the code is executed inside a critical section i_disable()/i_restore().
     */
    mask = _Active->td_sigpend & _Active->td_sigmask;
    if (mask != 0U)
    {   /* we have a signal waiting, return indicating an interrupted operation */
        int signo = find_first_bit(mask);
        _Active->td_sigpend &= ~((BitMask)1 << (BitMask)signo);
        i_restore(ps);
        if (outport != NULL)
            *outport = signo;

        if (signo == SIGCONT)
            return TIMEOUT_ID;

        return INTERRUPT_ID;
    }

    mask = rd->td_portpend & rd->td_portmask;
    if (mask != 0U)
    {                             /* we have a message waiting */
        Thread *sender;
        struct Gqueue *sendq;
        unsigned int len;
        int port;
        char *srqst;

        /* This is true 90% of the time, why not make it simple? */
        if (rd->td_portmask == 1U)
            port = 0;
        /* This implements a fair algo for port selection. */
        else
        {
            port = rd->td_lastport + 1;
            if ((port < rd->td_nports) &&
               ((mask >> (BitMask)port) != 0U))
                port += find_first_bit (mask >> (BitMask)port);
            else
                port = find_first_bit(mask);
    }

        rd->td_lastport = port;
        sendq = &rd->td_sendq[rd->td_lastport];
        sender = (Thread *)sendq->q_head;

        #ifdef SAM
        if (sender == (Thread *)sendq)
        {   /* Somebody died while they were on our queue */
            rd->td_portpend &= ~((BitMask)1 << (BitMask)port);
            i_restore(ps);
            goto again;
        }
        #endif

        /* Move sender from send queue to tail of reply queue. */
        sender->td_state = REPLY_BLOCKED;
        movequeue((struct Gqueue *)sender, &rd->td_replyq);

        /* If we removed the last sender from the queue we must clear the
           appropriate bit in the port mask. */
        if (sendq->q_head == sendq)
            rd->td_portpend &= ~((BitMask)1 << (BitMask)port);

        /* Copy the request message. */
        srqst = (char *)mp_ext2int(sender->td_rqst);
        len = MSG_SIZE (srqst);
        if (len > MSG_SIZE (rqst))
            len = MSG_SIZE(rqst);
        /* This works even for TMS320Cxx */
        if (len < sizeof(short))
            len = (unsigned int)sizeof(short);

        _Active->td_correspondent = sender->td_id;

        /* restore interrupts during the copy */
        i_restore(ps);
        (void)safe_copy(rqst, srqst, len);
        ps = i_disable();

#ifdef SAM
        if (_Active->td_correspondent == NULL)
            /* sender died */
            goto again;
#endif

        if (outport != NULL)
            *outport = port;
        i_restore (ps);
        return _Active->td_correspondent;
    }

    if (timeout == FOREVER)
    {
        _Active->td_state = RCV_BLOCKED;
        remqueue((struct Gqueue *)_Active);
        _Relinquish(ps);
        goto again;
    }

    /* Polling only? */
    if ((!timeout) || ((timeout->tv_sec == 0) && (timeout->tv_nsec == 0)))
    {   /* timed out */
        errno = ETIMEDOUT;
        i_restore(ps);
        return 0;
    }

    /* Check parameter */
    if ((timeout->tv_sec < 0) || (timeout->tv_nsec < 0) || (timeout->tv_nsec >= NANOSECOND))
    {
        errno = ETIMEDOUT;
        i_restore(ps);
        return 0;
    }

    sleeper.it_value = *timeout;
    sleeper.it_interval.tv_nsec = 0;
    sleeper.it_interval.tv_sec = 0;
    _Active->td_timeout.event.sigev_notify = SIGEV_SIGPOST; // timer_tick() must case SIGEV_SIGPOST
    _Active->td_timeout.event.sigev_signo = SIGCONT; // fix - which signo send?

    __timer_settime((TMCB *)(&_Active->td_timeout), 0, &sleeper, NULL);

    /* put ourself to sleep */
    _Active->td_state = TIMER_BLOCKED; /* indicate to mr_sigpost we're blocked on a timer */
    remqueue((struct Gqueue *)_Active);
    _Relinquish(ps);

    // reset signal event to default (for timeout functions)
    _Active->td_timeout.event.sigev_notify = SIGEV_NONE;
    _Active->td_timeout.event.sigev_signo = 0;
    goto again;

    /*
     * Here, some of compilers can generate a warning like "statement is unreachable".
     * But without this code some of them generate a warning like "no return, in function returning non-void".
     */
    return 0;   /* Not reachable */
}

/**
 * @} end rendezvous group
 */
