/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_pthread.h"
#include "../port/platform.h"


/**
 * @addtogroup pthread_group pthread
 * @{
 */


/**
 *  @brief Thread creation
 *
 *  @param [in] new_thread: pointer to a variable in which the thread-id will be placed.
 *  @param [in] attr: pointer to on thread attributes
 *  @param [in] start_routine: pointer to a function that will be executed within the created control flow.
 *  @param [in] arg: pointer to a single argument to the start_routine function.
 *
 *  @retval      EINVAL: error. not valid parameter.
 *  @retval      ENOMEM: error. not enough memory was available to create the new thread.
 *  @retval      0: success.
 *
 */
int pthread_create(pthread_t * new_thread, const pthread_attr_t * attr,
                void *(*start_routine) (void *arg), void *arg)
{
    register Thread *td;
    Status ps;
    int alloced = 0;
    void *stack_base;
    size_t stack_size;
    int priority;
    int detachstate;

    if (attr == NULL)
    {
        stack_base = NULL;
        stack_size = 0;
        priority = 0;
        detachstate = PTHREAD_CREATE_JOINABLE;
    }
    else if (PTHREAD_ATTR(attr) != NULL)
    {
        stack_base = PTHREAD_ATTR (attr)->stackbase;
        stack_size = PTHREAD_ATTR (attr)->stacksize;
        priority = PTHREAD_ATTR (attr)->priority;
        detachstate = PTHREAD_ATTR (attr)->detachstate;
    }
    else
        return EINVAL;

    if (stack_base == NULL)
    {                             /* we must allocate the stack */
        if (stack_size == 0U)
            stack_size = pthreadStackDefault;
        if (stack_size < platform_get_stack_min())
            return EINVAL;
        stack_base = kalloc(KA_STACK, stack_size);
        if (stack_base == NULL)
            return ENOMEM;
        alloced = 1;
        }
        else if (stack_size < platform_get_stack_min())
            return EINVAL;

    if (priority == 0)
    {
        /* use priority of calling thread */
        uint tmp = (uint)_Active->td_myqueue - (uint)_Ready_q;    // get distance between current queue address and top of queues address
        tmp = tmp / (uint)sizeof(struct Gqueue);                  // get priority (each queue is struct Gqueue)
        priority = (int)tmp;
    }

    td = _Init_thread (stack_base, stack_size, priority, arg, start_routine);
    if (td == NULL)
    {
        if (alloced != 0)
        kfree(KA_STACK, stack_base);
        return ENOMEM;
    }

    // Log this kernel function
    DLOG_PTHREAD_CREATE_INSERT;

    if (new_thread != NULL)
        *new_thread = td->td_id ;

    td->detachstate = detachstate;
    td->cancelstatus = 0;
    td->cleanup_stack = NULL;

    /* Make the new thread ready to run. */
    ps = i_disable();
    _Add_ready (td);
    _Relinquish (ps);

    return 0;
}


/**
 * @} end pthread group
 */
