/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_pthread.h"

void join_cleanup(void *arg);

/**
 * @addtogroup pthread_group pthread
 * @{
 */

/**
 *  @brief Join at termination
 *
 *  @param [in] thread: thread identificator.
 *  @param [in] value_ptr: pointer to a variable in which the thread's ending status will be placed.
 *
 *  @retval     0: success.
 *  @retval     ESRCH: error. thread can not be found.
 *  @retval     EDEADLK: error. value of the thread refers to the calling thread.
 *  @retval     EINVAL: error. not valid parameter.
 */
int pthread_join(pthread_t thread, void **value_ptr)
{
    Status ps;
    Thread *td;
    struct _pthread_cleanup_handler handler;
    int oldtype;

    /* Check for cancellation */
    CANCELLATION_POINT();

    // check valid of thread
    if (thread == (pthread_t)0)
    {
        return ESRCH;
    }

    if (pthread_equal(thread, pthread_self()) != 0)
    {
        return EDEADLK;
    }

    ps = i_disable();

    // find the thread
    td = ConvertToTD(thread);
    if (td == NULL)
    {
        i_restore(ps);
        return ESRCH;
    }

    // check thread to jonaible and empty join queue
    if ((td->detachstate != PTHREAD_CREATE_JOINABLE) || !queueIsEmpty((struct Gqueue *)&td->join_queue))
    {
        i_restore(ps);
        return EINVAL;
    }

    // thread already ending, and leave result
    if (td->td_state == DEAD)
    {
        if (value_ptr != NULL)
        {
            *value_ptr = td->exit_return;
        }
        td->td_id = UINT_MAX;
        TDfree(td);
        i_restore(ps);
        return 0;
    }

    // Log this kernel function
    DLOG_PTHREAD_JOIN_INSERT;

    // prepare for cancellation cleanup
    /* __pthread_cleanup_push(join_cleanup, (void*)td, &handler); */
    handler.routine = join_cleanup;
    handler.arg = td;
    handler.prev = (struct _pthread_cleanup_handler *)_Active->cleanup_stack;
    _Active->cleanup_stack = (void*)&handler;

    /* put myself on the wait queue */
    _Active->td_state = JOIN_BLOCKED;
    movequeue((struct Gqueue *)_Active, (struct Gqueue *) &td->join_queue);
    __pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);    //set async type
    _Relinquish(ps);
    __pthread_setcanceltype(oldtype, NULL);    //restore old type

    // remove cancellation handler
    /* __pthread_cleanup_pop(0, &handler); */
    _Active->cleanup_stack = (void*)handler.prev;

    if (value_ptr != NULL)
    {
        *value_ptr = td->exit_return;
    }
    td->td_id = UINT_MAX;
    TDfree(td);
    return 0;
}

/**
 * @} end pthread group
 */


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


/**
 *  @brief Join cleanup
 *
 *  @param [in] arg: pointer to function argument (thread identificator).
 *
 *  @retval no return value.
 */
void join_cleanup(void *arg)
{
    Thread *td = (Thread *)arg;

    // reset join queue
    initqueue(&(td->join_queue));
}


/**
 * @} end private kernel functions group
 */
