/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "_semaphore.h"
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include <stdarg.h>

#define EXCLUSIVE(oflag)    ((oflag) & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)

SCB *sHead = NULL;

/**
 * @addtogroup semaphore_group semaphore
 * @{
 */


/**
 *  @brief     Initialize/open a named semaphore
 *
 *  @param[in] name: pointer to a string naming a semaphore object.
 *  @param[in] oflag: argument controls whether the semaphore is created.
 *  @param[in] args: list of parameters.
 *
 *  @retval   sem_t: the address of the semaphore.
 *  @retval   sem_t(-1): error. Variable errno is set appropriately. (See errno.h)
 */
sem_t *sem_open(const char *name, int oflag, ...)
{
    SCB *scb;
    Status ps;
    size_t len;

    len = strlen(name);
    if ((len > (size_t)NAME_MAX) || (len > (size_t)PATH_MAX))
    {
        errno = ENAMETOOLONG;
        return SEM_FAILED;
    }

    // Log this kernel function
    DLOG_SEM_OPEN_INSERT;

    ps = i_disable();
    for (scb = sHead; scb != NULL; scb = scb->next)
        if (strcmp(scb->name, name) == 0)
      break;

    /* semaphore exists */
    if (scb != NULL)
    {
        if ((scb->flags & SEM_UNLINK) != 0U)
        {
            errno = EACCES;
            goto error;
        }

        if (EXCLUSIVE(oflag))
        {
            errno = EEXIST;
            goto error;
        }
        ++scb->ocount;
        i_restore(ps);
        return &scb->sem;
    }

    /* semaphore does not exist */
    if ((oflag & O_CREAT) != 0)
    {                             /* create it */
        /* must handle the var args */
        mode_t mode;    // ignored
        unsigned int value;
        va_list ap;

        va_start(ap, oflag);
        mode = va_arg(ap, mode_t);
        (void)mode;
        value = va_arg(ap, unsigned int);
        va_end(ap);

        if (value > (unsigned int)SEM_VALUE_MAX)
        {
            errno = EINVAL;
            goto error;
        }

        /* allocate the scb */
        scb = kalloc(KA_SEM, sizeof (SCB) + strlen (name));
        if (scb == NULL)
        {
            errno = ENOSPC;
            goto error;
        }

        /* init scb */
        (void)strcpy(scb->name, name);
        scb->sem.value = value;
        scb->ocount = 1;
        scb->flags = 0;
        initqueue((struct Gqueue *)&scb->sem.queue);
        /* link it in */
        scb->next = sHead;
        sHead = scb;
        i_restore(ps);
        scb->sem.id = (unsigned int)&scb->sem;
        return &scb->sem;
    }

    /* semaphore does not exist */
    errno = ENOENT;

error:
    i_restore(ps);
    return SEM_FAILED;
}


/**
 * @} end semaphore group
 */
