/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "idle.h"
#include <bits/local_lim.h>
#include "../port/platform.h"


/* Stack variables visible to the user */
size_t pthreadStackDefault;
int pthreadStackFill = 0;

struct Gqueue _Ready_q[PTHREAD_PRIO_MAX + 1];
struct Gqueue *prio_q;

int errno = 0;

static pthread_t CreateSysThread (void *(*f) (void *arg), size_t stksz, int prio, unsigned long fatal);


/**
 * @addtogroup misc_group misc
 * @{
 */

/**
 *  @brief     Finishes initialization and starts the first user thread.
 *
 *
 *  @return    none.
 */
void Nano_Start(void)
{
    int i;

    (void)i_disable();  // disable interrupts just in case. without a matching
                // i_enable, interrupts will be disabled throught startup
                // after startup, their will be a corresponding enable as
                // the last thing before dispatch.  KR


    /* initialize the priority queues */
    for (i = 0; i < (PTHREAD_PRIO_MAX + 1); ++i)
        initqueue ((struct Gqueue *)&_Ready_q[i]);

    pthreadStackDefault = platform_get_stack_default();
    if (_IdleThreadStack == 0U)
        _IdleThreadStack = platform_get_stack_min();

    /* Create idle thread */
    _IdleID = CreateSysThread (_IdleThread, _IdleThreadStack, 0, FAT_IDLE);

    /* Create root thread */
    if ((ConfigTable.rootPriority <= 0) ||
        (ConfigTable.rootPriority > PTHREAD_PRIO_MAX))
    {
        kfatal (FAT_ROOT);
    }

    (void)CreateSysThread (ConfigTable.rootStartRoutine,
                   ConfigTable.rootStackSize,
                   ConfigTable.rootPriority, FAT_ROOT);

    _isrlevel = 0;
    _locklevel = 0;

    // interrupts can not be enabled here because dispatch calls relinquish
    // and must be called disabled.
    _Dispatch ();
}

/**
 * @} end misc group
 */


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */

/**
 *  @brief     Initializes a stack.
 *
 *  @param[in] stack: points to the start of the allocated portion of the stack.
 *  @param[in] stacksize: size of stack.
 *  @param[in] priority: thread priority.
 *  @param[in] arg: root function argument.
 *  @param[in] root_func: pointer to root function.
 *
 *  @return    Pointer to object thread stack or NULL if memory didn't allocate.
 */
Thread *_Init_thread(char *stack, size_t stacksize,
              int priority, void *arg, THREAD (*root_func)(void *arg))
{
    register Thread *td;
    char * sp;

    td = TDalloc(priority);
    if (td == NULL)
        return NULL;

    td->td_stackbase = stack;
    td->td_stacksize = stacksize;

    td->td_myqueue = (struct Gqueue *)&_Ready_q[priority];
    td->td_rootfunc = root_func;

    if (pthreadStackFill != 0)
        (void)memset(stack, (int)'S', stacksize);


    // Init stack for thread start
    sp = platform_init_thread (td, arg, root_func);


    /* create the initialization for the timer so that it can
     be used without a timer_create call because its built in */
    td->td_timeout.timerid = (timer_t)&(td->td_timeout);
    td->td_timeout.td = td;

    /* Set up the Q so remqueue() doesn't barf */
    initqueue(((struct Gqueue *)&(td->td_timeout.queue)));


    /* set the inital stack pointer */
    td->td_stackpointer = sp;

    initqueue(&(td->join_queue));     // for saving join-block thread

#ifdef DLOG_SUPPORT
    td->dlog_mask = 0;
#endif

    return td;
}


/**
 *  @brief     Create idle thread.
 *
 *  @param[in] f: pointer to thread function.
 *  @param[in] stksz: Description for stksz
 *  @param[in] prio: thread priority.
 *  @param[in] fatal: fills KFATAL.
 *
 *  @return    idle thread identifier.
 */
static pthread_t CreateSysThread(void*(*f)(void *arg), size_t stksz, int prio, unsigned long fatal)
{
    size_t stack_size;
    register Thread *td;
    register char *stack;

    if (stksz == 0U)
        stack_size = pthreadStackDefault;
    else if (stksz < (size_t)platform_get_stack_min())
        stack_size = (size_t)platform_get_stack_min();
    else
        stack_size = stksz;
    stack = kalloc(KA_STACK, stack_size);

    if (stack == NULL)
        kfatal(fatal);
    td = _Init_thread(stack, stack_size, prio, NULL, f);

    if (td == NULL)
        kfatal(fatal);
    _Add_ready(td);

    return td->td_id;
}


/**
 *  @brief     This routine adds a thread to it's ready queue.
 *
 *  @param[in] td: pointer to the thread structure object.
 *
 *  @return    none.
 */
void _Add_ready(Thread *td)
{
    td->td_state = READY;

    /* The first time _Add_ready happens prio_q points to 0
     initialization will be handled by assignment in this if statement
     hopefully to the start of the _Ready_q */

    if ((uint_32) td->td_myqueue > (uint_32) prio_q)
        prio_q = td->td_myqueue;

    insqueue((struct Gqueue *)td, (struct Gqueue *)td->td_myqueue);
}

/**
 *  @brief     Make context switch.
 *
 *  @return    none.
 */
void ContextSwitch(void)
{
    /* Hunt for the next non-empty queue. Guaranteed to succeed because the
       idle task lives in the lowest priority queue. */
    while (queueIsEmpty (prio_q))
    {
        prio_q--;
    }

    /* Make the head of the queue active. */
    _Active = (Thread *)prio_q->q_head;

    // Log thread running
    DLOG_PTHREAD_RUN_INSERT;
}



#ifdef DLOG_SUPPORT
/**
 *  @brief     Set logging events mask.
 *
 *  @param[in] id: thread identifier.
 *  @param[in] mask: mask value.
 *
 *  @retval    0: success.
 *  @retval    -1: thread identifier doesn't exist.
 */
int dlog_thread_mask(pthread_t id, uint_32 mask)
{
    Thread *td;

    td = ConvertToTD(id);
    if (td == NULL)
        return -1;

    td->dlog_mask = mask;
    return 0;
}
#endif

/**
 * @} end private kernel functions group
 */
