/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include "dlog_kernel_ev.h"
#include "kernel.h"
#include "_time.h"


#if _POSIX_TIMERS > 0


/**
 * @addtogroup timer_group timer
 * @{
 */


/**
 *  @brief     Set time of the timer.
 *
 *  @param[in] timerid: identificator of the timer.
 *  @param[in] flags: specific flag for time type (TIMER_ABSTIME or TIMER_REALTIME).
 *  @param[in] value: pointer to value of timer.
 *  @param[in] ovalue: pointer to value of timer.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
int timer_settime(timer_t timerid, int flags, const struct itimerspec *value, struct itimerspec *ovalue)
{
    if (!VALID_TIMERID (timerid))
        RETURN_ERRNO(EINVAL);

    if ((value->it_value.tv_nsec < 0) || (value->it_interval.tv_nsec < 0))
        RETURN_ERRNO(EINVAL);

    if ((value->it_value.tv_sec < 0) || (value->it_interval.tv_sec < 0))
        RETURN_ERRNO(EINVAL);

    if ((value->it_value.tv_nsec >= NANOSECOND) || (value->it_interval.tv_nsec >= NANOSECOND))
        RETURN_ERRNO(EINVAL);

    // Log this kernel function
    DLOG_TIMER_SETTIME_INSERT;

    __timer_settime((TMCB *)timerid, flags, value, ovalue);

    return 0;
}


/**
 * @} end timer group
 */


/**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


 /**
 *  @brief     Set time of the timer.
 *
 *  @param[in] tmcb: pointer to timer control object.
 *  @param[in] flags: specific flag for time type (TIMER_ABSTIME or TIMER_REALTIME).
 *  @param[in] value: pointer to value of timer.
 *  @param[in] ovalue: pointer to value of timer.
 *
 *  @return    no return value.
 */
void __timer_settime(TMCB *tmcb, int flags, const struct itimerspec *value, struct itimerspec *ovalue)
{
    Status ps;

    if (ovalue != NULL)
        __timer_gettime(tmcb, ovalue);

    if ((value == NULL) || (ZeroTime (&value->it_value)))
    {                             /* disarm timer */
        ps = i_disable();
        remqueue((struct Gqueue *)&tmcb->queue);
        tmcb->state &= ~TIMER_ARMED;
        i_restore(ps);
        return;
    }

    tmcb->timeout = value->it_value;
    tmcb->period = value->it_interval;

    // If the timer was already armed, need to remove it before to add
    if ((tmcb->state & TIMER_ARMED) != 0U)
    {
        ps = i_disable();
        remqueue((struct Gqueue *)&tmcb->queue);
        i_restore(ps);
    }
    else
        tmcb->state |= TIMER_ARMED;

    ps = i_disable();
    if ((flags & TIMER_ABSTIME) == 0)
    {                             /* add in the current time */
        register struct timespec *abstime;
        abstime = &tmcb->timeout;
        abstime->tv_sec += CurrentTime.tv_sec;
        abstime->tv_nsec += CurrentTime.tv_nsec;
        while (abstime->tv_nsec >= NANOSECOND)
        {
            abstime->tv_nsec -= NANOSECOND;
            abstime->tv_sec++;
        }
    }
    else
        tmcb->state |= TIMER_ABS;

    /* insert into the timer queue */
    insertTimerQueue(tmcb);
    i_restore(ps);
}


/**
 * @} end private kernel functions group
 */


#endif /* _POSIX_TIMERS */
