/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include <sys.h>
#include <mrendez.h>
#include "kernel.h"
#include "_time.h"
#include "threads_utilization.h"


/* DSPnano extension to the Posix timers option. This is the tick routine
   called by the hardware ISR when a clock tick happens. It handles expiry of
   timers, and timeouts. */
#if _POSIX_TIMERS > 0
struct timespec CurrentTime = { 0, 0 };
struct timespec UpTime = { 1, 0 };

/* The timer queue sentinel. */
struct Gqueue TimerQ = {
  &TimerQ,
  &TimerQ
};


/* Since this is where it is used - this is where it is defined. Defaults to
   20ms. */
long _Timer_period = _POSIX_CLOCKRES_MIN;  // use 'long' to align with other funcs


/**
 * @addtogroup timer_group timer
 * @{
 */


/**
 *  @brief     Kernel ISR handler.
 *
 *  @param[in] clockid: specified clock ID.
 *
 *  @retval    0: success.
 *  @retval   -1: error. Variable errno is set appropriately. (See errno.h)
 */
void timer_tick(clockid_t clockid)
{
    Status ps;

    if (clockid != CLOCK_REALTIME)
        return;

    ps = i_disable();

    /* Threads utilization measurement */
    if (_threads_count != 0)
    {
        int i;

        for (i = 0; i < _threads_count; i++)
        {
            if (_threads_table[i].thread_id == (unsigned int)_Active->td_id)
                _threads_table[i].util++;
        }
    }

    /* Up time calculate */
    UpTime.tv_nsec += _Timer_period;

    /* Done with a loop to avoid division. Since it is unlikely that
       _Timer_period is greater than 1 second, normally the loop will be
       entered at most once. */
    while (UpTime.tv_nsec >= NANOSECOND)
    {
        UpTime.tv_nsec -= NANOSECOND;
        UpTime.tv_sec++;
    }

    /* Current time calculate */
    CurrentTime.tv_nsec += _Timer_period;

    /* Done with a loop to avoid division. Since it is unlikely that
       _Timer_period is greater than 1 second, normally the loop will be
       entered at most once. */
    while (CurrentTime.tv_nsec >= NANOSECOND)
    {
        CurrentTime.tv_nsec -= NANOSECOND;
        CurrentTime.tv_sec++;
    }

    /* Process the timer queue if necessary */
    while (!queueIsEmpty(&TimerQ) && timer_le(((TMCB *)TimerQ.q_head)->timeout, CurrentTime))
    {
        register TMCB *tmcb;

        tmcb = (TMCB *)TimerQ.q_head;
        /* remove from the timer queue */
        remqueue((struct Gqueue *)tmcb);
        tmcb->state &= ~TIMER_ARMED;

        switch (tmcb->event.sigev_notify)
        {
            case SIGEV_NONE:
                /* Just ready the thread and set its state to TIMER_EXPIRED. */
                remqueue((struct Gqueue *)tmcb->td);
                _Add_ready(tmcb->td);
                tmcb->td->td_state = TIMER_EXPIRED;
                break;

            case SIGEV_SIGPOST:
                /* The sigpost will ready the thread if necessary */
                (void)mr_sigpost(tmcb->td->td_id, tmcb->event.sigev_signo);
                break;

            default:
                break;
        }

        if (!ZeroTime(&tmcb->period))
        {                         /* add the period and requeue */
            tmcb->timeout.tv_sec += tmcb->period.tv_sec;
            tmcb->timeout.tv_nsec += tmcb->period.tv_nsec;
            if (tmcb->timeout.tv_nsec >= NANOSECOND)
            {
                tmcb->timeout.tv_nsec -= NANOSECOND;
                tmcb->timeout.tv_sec++;
            }
            tmcb->state |= TIMER_ARMED;
            insertTimerQueue(tmcb);
        }
    }

    _Relinquish(ps);
}


/**
 * @} end timer group
 */


 /**
 * @addtogroup private_kernel private kernel functions
 * @{
 */


 /**
 *  @brief     Insert tmcb into the TimerQ in sorted order.
 *
 *  @param[in] tmcb: pointer to timer control object.
 *
 *  @return    no return value.
 */
void insertTimerQueue(TMCB * tmcb)
{
    register struct Gqueue *queue;
    queue = (struct Gqueue *)TimerQ.q_head;

    while ((queue != &TimerQ) && !timer_lt(tmcb->timeout, ((TMCB *)queue)->timeout))
        queue = (struct Gqueue *)queue->q_head;

    insqueue ((struct Gqueue *)tmcb, queue);
}


/**
 * @} end private kernel functions group
 */


#endif /* _POSIX_TIMERS */
