/*
*********************************************************************************************************
*                  Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS, VehicleOS
*                                            for STM32Cube
*
*                           (c) Copyright 2007 - 2018; RoweBots Research Inc.
*                                           www.rowebots.com
*********************************************************************************************************
* Licensing:
*          YOUR USE OF THIS SOFTWARE IS SUBJECT TO THE TERMS OF A ROWEBOTS SOFTWARE LICENSE.
* 1. If you are not willing to accept the terms of an appropriate RoweBots Software License, you must
*    not download or use this software for any reason.
* 2. Information about RoweBots licensing regarding this software available at
*    www.rowebots.com/products/licensing/st-v1.
* 3. It is your obligation to select an appropriate license based on your intended use of the
*    Unison OS, Unison RTOS, WearableOS, MedicalOS, ConsumerOS or VehicleOS.
* 4. Unless you have executed a RoweBots Source Code License Agreement, your use of the RoweBots OS
*    offering is limited to evaluation, educational or personal non-commercial uses.
* 5. If you use this software package with RoweBots OS components for STMicroelectronics MCU/MPU, you
*    must still execute a RoweBots Source Code License Agreement and abide by the terms of this License
*    with the caveat that fees for commercial use are waived for those components.
* 6. The Unison OS and its' variants may not be redistributed or disclosed to any third party without
*    the written consent of RoweBots Research Inc. Only STMicroelectronics can redistribute this source
*    code, in case it would be reused in one package, through the SLA0048 license: www.st.com/SLA0048.
*********************************************************************************************************
* Documentation and Working Examples:
*    You can find user manuals, API references and more at www.rowebots.com. All provided source code
*    examples are in ready to run state.
*********************************************************************************************************
* Technical Support:
*    Support is available for all users of RoweBots software. For additional information on support, see
*    www.rowebots.com/support or you can contact support@rowebots.com.
*********************************************************************************************************
*/

#include "port.h"


extern void ContextSwitch(void);


/**
 * @addtogroup port_group port
 * @{
 */


/**
 *  @brief     Interrupt handler for first task starting.
 *             Called from StartFirstTask().
 *
 *  @param     none.
 *
 *  @return    no return value.
 */
void StartFirstTask_Handler(void) __attribute__ (( naked ));
void StartFirstTask_Handler(void)
{
#if defined(__ARMCORTEXM0__)
    __asm volatile ("ldr        r0, =_Active");
    __asm volatile ("ldr        r2, [r0]");
    __asm volatile ("ldr        r0, [r2, #8]");        //Stack offset in structure

    __asm volatile ("add        r0, r0, #16");
    __asm volatile ("ldmia      r0!, {r4-r7}");        //Load the high registers
    __asm volatile ("mov        r8, r4");            //Store the high registers
    __asm volatile ("mov        r9, r5");
    __asm volatile ("mov        r10, r6");
    __asm volatile ("mov        r11, r7");
    __asm volatile ("ldmia      r0!, {r4-r5}");        //Load R14 and Int status
    __asm volatile ("mov        r14, r4");
    __asm volatile ("msr        primask, r5");        //Set int status

    __asm volatile ("msr        psp, r0");            //Write R0 to Process Stack Pointer

    __asm volatile ("sub        r0, r0, #40");        //Go back for the low registers that are not automatically restored.
    __asm volatile ("ldmia      r0!, {r4-r7}");        //Load the low registers

    __asm volatile ("ldr        r1, [r2, #24]");    //errno offset in structure
    __asm volatile ("ldr        r0, =errno");
    __asm volatile ("str        r1, [r0]");

    //Return to Thread mode, uses PSP
    __asm volatile ("mov        r1, r14");    //OR R14 with 0x0d
    __asm volatile ("movs       r0, #0x0d");
    __asm volatile ("orr        r1, r0");
    __asm volatile ("bx         r1");

#elif defined(__ARMCORTEXM3__) || defined(__ARMCORTEXM4__) || defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    __asm volatile ("ldr        r0, =_Active");
    __asm volatile ("ldr        r2, [r0]");
    __asm volatile ("ldr        r0, [r2, #8]");        //Stack offset in structure

    __asm volatile ("ldmia      r0!, {r4-r11, r14}");    //Load registers

#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    //If the FPU context using, load high vfp registers
    __asm volatile ("tst        r14, #0x10");
    __asm volatile ("it         EQ");
    __asm volatile ("vldmiaeq   r0!,    {s16-s31}");
#endif

    __asm volatile ("ldmia      r0!, {r1}");        //Int status
    __asm volatile ("msr        primask, r1");        //Int enable

    __asm volatile ("msr        psp, r0");            //Write R0 to Process Stack Pointer

    __asm volatile ("ldr        r1, [r2, #24]");    //errno offset in structure
    __asm volatile ("ldr        r0, =errno");
    __asm volatile ("str        r1, [r0]");

    //Return to Thread mode, uses PSP
    //0xfffffffd - non-floating-point state
    //0xffffffed - floating-point state
    __asm volatile ("orr        r14, r14, #0xd");
    __asm volatile ("bx         r14");
#endif /* (__ARMCORTEXM3__ || __ARMCORTEXM4__ || __ARMCORTEXM4F__ || __ARMCORTEXM7__) */
}


/**
 *  @brief     This function makes context store/restore and tasks switch.
 *             This interrupt is called after all interrupts.
 *
 *  @param     none.
 *
 *  @return    no return value.
 */
void TaskSwitch_Handler(void)  __attribute__ (( naked ));
void TaskSwitch_Handler(void)
{
    __asm volatile ("cpsid    i");    //Disable interrupt

#if defined(__ARMCORTEXM0__)
    /**********************/
    /* Store task context */
    /**********************/
    __asm volatile ("mrs        r1, psp");          //Get task stack

    __asm volatile ("ldr        r2, =_Active");
    __asm volatile ("ldr        r2, [r2]");

    __asm volatile ("ldr        r0, =errno");
    __asm volatile ("ldr        r0, [r0]");
    __asm volatile ("str        r0, [r2, #24]");    //errno offset in structure

    __asm volatile ("sub        r1, r1, #40");      //Adjust stack pointer to where memory needs to be stored
                                                    //40 bytes = r4-r11 + r14 + 4 bytes Int status, default increments SP after each storing
    __asm volatile ("stm        r1!, {r4-r7}");     //Save rest registers
    __asm volatile ("mov        r4, r8");           //Store the high registers
    __asm volatile ("mov        r5, r9");
    __asm volatile ("mov        r6, r10");
    __asm volatile ("mov        r7, r11");
    __asm volatile ("stm        r1!, {r4-r7}");     //Save rest registers

    __asm volatile ("mov        r4, r14");
    __asm volatile ("mov        r5, #0");           //Int status =1, if we in _Relinquish
    __asm volatile ("stm        r1!, {r4-r5}");     //Save R14 and Int status
    __asm volatile ("sub        r1, r1, #40");      //CM0+ does not automatically call back the SP to initial location so we must do this manually

    __asm volatile ("str        r1, [r2, #8]");     //Save new stack


    //Change task
    ContextSwitch();


    /*************************/
    /* Load new task context */
    /*************************/
    __asm volatile ("ldr        r0, =_Active");
    __asm volatile ("ldr        r2, [r0]");
    __asm volatile ("ldr        r0, [r2, #8]");     //Stack offset in structure

    __asm volatile ("add        r0, r0, #16");
    __asm volatile ("ldmia      r0!, {r4-r7}");     //Load the high registers
    __asm volatile ("mov        r8, r4");           //Store the high registers
    __asm volatile ("mov        r9, r5");
    __asm volatile ("mov        r10, r6");
    __asm volatile ("mov        r11, r7");
    __asm volatile ("ldmia      r0!, {r4-r5}");     //Load R14 and Int status
    __asm volatile ("mov        r14, r4");
    __asm volatile ("msr        primask, r5");      //Set int status

    __asm volatile ("msr        psp, r0");          //Write R0 to Process Stack Pointer

    __asm volatile ("sub        r0, r0, #40");      //Go back for the low registers that are not automatically restored.
    __asm volatile ("ldmia      r0!, {r4-r7}");     //Load the low registers

    __asm volatile ("ldr        r1, [r2, #24]");    //errno offset in structure
    __asm volatile ("ldr        r0, =errno");
    __asm volatile ("str        r1, [r0]");

    //Return to Thread mode, uses PSP
    __asm volatile ("mov        r1, r14");          //OR R14 with 0x0d
    __asm volatile ("movs       r0, #0x0d");
    __asm volatile ("orr        r1, r0");
    __asm volatile ("bx         r1");

#elif defined(__ARMCORTEXM3__) || defined(__ARMCORTEXM4__) || defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    /**********************/
    /* Store task context */
    /**********************/
    __asm volatile ("mrs        r1, psp");          //Get task stack

    __asm volatile ("ldr        r2, =_Active");
    __asm volatile ("ldr        r2, [r2]");

    __asm volatile ("ldr        r0, =errno");
    __asm volatile ("ldr        r0, [r0]");
    __asm volatile ("str        r0, [r2, #24]");    //errno offset in structure

    __asm volatile ("mov        r0, #0");           //Int status =1, if we in _Relinquish
    __asm volatile ("stmdb      r1!, {r0}");        //Save int status

#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    //If the FPU context using, save high vfp registers
    __asm volatile ("tst        r14, #0x10");
    __asm volatile ("it         EQ");
    __asm volatile ("vstmdbeq   r1!,    {s16-s31}");
#endif

    __asm volatile ("stmdb      r1!, {r4-r11, r14}");   //Save rest registers
    __asm volatile ("str        r1, [r2, #8]");         //Save new stack


    //Change task
    ContextSwitch();


    /*************************/
    /* Load new task context */
    /*************************/
    __asm volatile ("ldr        r0, =_Active");
    __asm volatile ("ldr        r2, [r0]");
    __asm volatile ("ldr        r0, [r2, #8]");         //Stack offset in structure

    __asm volatile ("ldmia      r0!, {r4-r11, r14}");   //Load registers

#if defined(__ARMCORTEXM4F__) || defined(__ARMCORTEXM7__)
    //If the FPU context using, load high vfp registers
    __asm volatile ("tst        r14, #0x10");
    __asm volatile ("it         EQ");
    __asm volatile ("vldmiaeq   r0!,    {s16-s31}");
#endif

    __asm volatile ("ldmia      r0!, {r1}");        //Load int status
    __asm volatile ("msr        primask, r1");      //Set int status

    __asm volatile ("msr        psp, r0");          //Write R0 to Process Stack Pointer

    __asm volatile ("ldr        r1, [r2, #24]");    //errno offset in structure
    __asm volatile ("ldr        r0, =errno");
    __asm volatile ("str        r1, [r0]");

    //Return to Thread mode, uses PSP
    //0xfffffffd - non-floating-point state
    //0xffffffed - floating-point state
    __asm volatile ("orr        r14, r14, #0xd");
    __asm volatile ("bx         r14");
#endif /* (__ARMCORTEXM3__ || __ARMCORTEXM4__ || __ARMCORTEXM4F__ || __ARMCORTEXM7__) */
}


/**
 *  @brief     Start first RTOS thread
 *
 *  @param     none.
 *
 *  @return    no return value.
 */
void StartFirstTask (void) __attribute__ (( naked ));
void StartFirstTask (void)
{
    __asm volatile ("ldr    r0, =0xE000ED08");  //Vector Table Offset Register
    __asm volatile ("ldr    r0, [r0]");         //Pointer to stack
    __asm volatile ("ldr    r0, [r0]");         //Stack
    __asm volatile ("msr    msp, r0");          //Write R0 to Main Stack Pointer
    __asm volatile ("cpsie  i");                //Enable interrupts
    __asm volatile ("svc    0");                //Interrupt for start first task
}


/**
 * @} end port functions group
 */
